import React, { useState, useEffect } from 'react';
import { useHistory, useParams } from 'react-router-dom';
import Cookies from 'js-cookie';
import {
  Layout,
  CoreText,
  TextType,
  StyledLayout,
  FormGroup,
  FormGroupOrientation,
  Input,
  InputType,
  Select,
  Button,
  ButtonType,
  Display,
  JustifyContent,
  ButtonState,
  Background,
  BorderRadius,
  AlignItems
} from 'twitch-core-ui';
import DatePicker from './date-picker';
import Loading from './loading';
import ErrorAlert from './error-alert';
import { ATLAS_BASE_URL } from '../utils/config';
import { useFetchPOST } from '../hooks/useFetch';
import { fetchWithTimeout } from '../utils/helpers';
import { useLocalStorage } from '@rehooks/local-storage';
import './styles/Form.scss';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function SeasonsEdit() {
  const history = useHistory();
  let { id } = useParams<{ id: string }>();
  let options = {
    limit: 1,
    page: 0,
    eventtype: 1,
    'req-cols': ['*'],
    filters: [
      {
        'filter-type': 1,
        key: 'season_id',
        v1: id
      }
    ]
  };
  const [data, loading] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-season-list`, options);
  const [isFormSubmitting, setIsFormSubmitting] = useState(false);
  const [error, setError] = useState('');

  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  const [seasonOptions] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-season-list`, {
    eventtype: 1,
    filters: [{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }],
    'req-cols': ['product_id', 'season_id', 'season_name', 'product_name']
  });
  const [productOptions] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-product-list`, {
    eventtype: 1,
    limit: 1000,
    'req-cols': ['product_id', 'product_name'],
    filters: [{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }],
    sortdata: [{ ord: 0, col: 'product_name', asc: true }]
  });

  // modal form data
  // hidden
  const [seasonId, setSeasonId] = useState('0');
  // page 1
  const [seasonName, setSeasonName] = useState('');
  const [description, setDescription] = useState('');
  const [productId, setProductId] = useState('0');
  const [startDate, setStartDate] = useState(null);
  const [endDate, setEndDate] = useState(null);
  const [projectedHoursBroadcast, setProjectedHoursBroadcast] = useState('');
  const [estimatedPrizePool, setEstimatedPrizePool] = useState('');
  const [precedingSeason, setPrecedingSeason] = useState('');
  const [, setPrecedingSeasonName] = useState('');
  const [precedingSeasonOptions, setPrecendingSeasonOptions] = useState([]);

  const handleProductChange = productId => {
    setProductId(productId);
    if (seasonOptions && seasonOptions.data) {
      let precedingSeasons = seasonOptions.data.items.filter(item => item.product_id === parseInt(productId));
      setPrecendingSeasonOptions(precedingSeasons);
    }
  };

  useEffect(() => {
    if (data !== undefined && data.data !== undefined) {
      const {
        season_id,
        season_name,
        description,
        prize_pool,
        pledged_hours_broadcast,
        start_time,
        end_time,
        product_id,
        previous_season_id
      } = data.data.items[0];

      setSeasonId(season_id || '0');
      setSeasonName(season_name || '');
      setDescription(description || '');
      setEstimatedPrizePool(prize_pool || '');
      setProjectedHoursBroadcast(pledged_hours_broadcast || '');
      setStartDate(start_time ? new Date(start_time) : null);
      setEndDate(end_time ? new Date(end_time) : null);
      handleProductChange(product_id || '0');
      setPrecedingSeason(previous_season_id || '0');
    } /* eslint-disable react-hooks/exhaustive-deps */
  }, [data]);

  const submitForm = () => {
    let season = {
      season_id: seasonId,
      season_name: seasonName,
      description: description,
      start_time: startDate.toISOString(),
      end_time: endDate.toISOString(),
      product_id: productId,
      pledged_hours_broadcast: projectedHoursBroadcast,
      prize_pool: estimatedPrizePool,
      previous_season_id: precedingSeason,
      is_active: true
    };

    let obj = season;

    const body = Object.keys(obj)
      .filter(e => obj[e] !== null)
      .reduce((o, e) => {
        o[e] = obj[e];
        return o;
      }, {});

    setIsFormSubmitting(true);
    async function fetchData() {
      let token = Cookies.get('token');
      const options = {
        method: 'POST',
        headers: {
          Accept: 'application/json',
          'Content-Type': 'application/json',
          Authorization: `Bearer ${token}`
        },
        body: JSON.stringify(body)
      };
      try {
        const response = await fetchWithTimeout(`${ATLAS_BASE_URL}/event/edit-season`, options);
        const json = await response.json();
        console.log(`/edit-season:`, json);

        if (json.success) {
          history.push('/seasons');
        } else {
          setError(!!json.error_message ? json.error_message : 'Edit Season request not successful');
          setIsFormSubmitting(false);
        }
      } catch (err) {
        console.error(`/edit-season:`, err);
        setError(err);
        setIsFormSubmitting(false);
      }
    }
    fetchData();
  };

  const isFormValid = !!seasonName && !!startDate && !!endDate;

  return (
    <Layout className="form--wrapper" margin="auto">
      <Layout padding={{ bottom: 1 }}>
        <CoreText type={TextType.H3}>Edit Season</CoreText>
      </Layout>
      <Layout margin={{ bottom: 2 }}>
        <CoreText type={TextType.H4}>Season Details</CoreText>
      </Layout>
      {loading ? (
        <Loading />
      ) : (
        <>
          <StyledLayout background={Background.Base} borderRadius={BorderRadius.Medium} border>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Name" hint="Name your season" required>
                <Input type={InputType.Text} placeholder="Season Name" value={seasonName} onChange={e => setSeasonName(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Product" hint="The product under which the season falls">
                <Select value={productId} onChange={e => handleProductChange(e.currentTarget.value)}>
                  <option disabled value="0">
                    Select Product
                  </option>
                  {productOptions &&
                    productOptions.data &&
                    productOptions.data.items.map(option => (
                      <option value={option.product_id} key={option.product_id}>
                        {option.product_name}
                      </option>
                    ))}
                </Select>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Description" hint="Describe this season">
                <Input type={InputType.Text} placeholder="Season Description" value={description} onChange={e => setDescription(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Start &amp; End Date" required>
                <Layout display={Display.Flex} alignItems={AlignItems.Center} fullWidth>
                  <Layout margin={{ right: 2 }} fullWidth>
                    <DatePicker defaultDate={startDate} onChange={setStartDate} maxDate={endDate} />
                  </Layout>
                  <Layout fullWidth>
                    <DatePicker defaultDate={endDate} onChange={setEndDate} minDate={startDate} position="right" />
                  </Layout>
                </Layout>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Estimated Hours Broadcast" hint="The projected hours broadcast of your season">
                <Input
                  type={InputType.Number}
                  placeholder="Hours Broadcast"
                  value={projectedHoursBroadcast}
                  onChange={e => setProjectedHoursBroadcast(e.currentTarget.value)}
                />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Estimated Prize Pool" hint="The projected prize pool of your season">
                <Input type={InputType.Number} placeholder="$XXX,XXX" value={estimatedPrizePool} onChange={e => setEstimatedPrizePool(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Preceding Season" hint="The preceding season of this current season">
                <Select
                  value={precedingSeason}
                  onChange={e => {
                    setPrecedingSeason(e.currentTarget.value);
                    setPrecedingSeasonName(e.currentTarget.innerText);
                  }}>
                  <option value="0" disabled>
                    Select Preceding Season
                  </option>
                  {precedingSeasonOptions &&
                    precedingSeasonOptions.map(option => (
                      <option key={option.season_id} value={option.season_id}>
                        {option.product_name} {option.season_name}
                      </option>
                    ))}
                </Select>
              </FormGroup>
            </StyledLayout>
          </StyledLayout>

          {error && (
            <Layout margin={{ y: 3 }}>
              <ErrorAlert error={error} />
            </Layout>
          )}

          <Layout margin={{ y: 3 }} display={Display.Flex} justifyContent={JustifyContent.End}>
            <Layout margin={{ right: 1 }}>
              <Button
                type={ButtonType.Secondary}
                linkTo="/seasons"
                state={isFormSubmitting ? ButtonState.Loading : ButtonState.Default}
                disabled={isFormSubmitting}>
                Cancel
              </Button>
            </Layout>
            <Button onClick={submitForm} state={isFormSubmitting ? ButtonState.Loading : ButtonState.Default} disabled={isFormSubmitting || !isFormValid}>
              Submit
            </Button>
          </Layout>
        </>
      )}
    </Layout>
  );
}

export default SeasonsEdit;
