import React from 'react';
import Cookies from 'js-cookie';
import {
  StyledLayout,
  Background,
  FontSize,
  Table,
  TableHeader,
  TableHeading,
  TableBody,
  TableRow,
  TableCell,
  Sorting,
  CoreText,
  Layout,
  Position
} from 'twitch-core-ui';
import MoreActions from './more-actions';
import { ATLAS_BASE_URL } from '../utils/config';
import { fetchWithTimeout } from '../utils/helpers';
import './styles/Table.scss';

function SeasonsList({ seasons, editSeason = undefined }) {
  const seasonsListHeaders = ['Product', 'Season', 'Expected Hours Broadcast', 'Estimated Prize Pool', 'Preceded By'];

  const deleteSeason = season => {
    let obj = {
      ...season,
      is_active: false
    };

    const body = Object.keys(obj)
      .filter(e => obj[e] !== null)
      .reduce((o, e) => {
        o[e] = obj[e];
        return o;
      }, {});

    async function fetchData() {
      let token = Cookies.get('token');
      const options = {
        method: 'POST',
        headers: {
          Accept: 'application/json',
          'Content-Type': 'application/json',
          Authorization: `Bearer ${token}`
        },
        body: JSON.stringify(body)
      };
      try {
        const response = await fetchWithTimeout(`${ATLAS_BASE_URL}/event/edit-season`, options);
        const json = await response.json();
        console.log(`/edit-season:`, json);
        document.location.reload();
      } catch (error) {
        console.error(`/edit-season:`, error);
        // TODO: on error display error message somewhere
      }
    }
    fetchData();
  };

  return (
    <StyledLayout className="seasons--card-wrapper table--wrapper" background={Background.Base} fontSize={FontSize.Size7} position={Position.Relative}>
      <Table alternateRows>
        <TableHeader>
          <TableRow>
            {seasonsListHeaders.map(tableHeader =>
              tableHeader === 'Event Name' ? (
                <TableHeading
                  sorting={Sorting.Default}
                  key={tableHeader}
                  label={tableHeader}
                  onClick={() => console.log(`[table-wrapper.tsx]: sort table header`)}
                />
              ) : (
                <TableHeading key={tableHeader}>
                  <CoreText ellipsis>{tableHeader}</CoreText>
                </TableHeading>
              )
            )}
          </TableRow>
        </TableHeader>
        <TableBody>
          {seasons &&
            seasons.map(season => {
              const { season_id, product_name, season_name, pledged_hours_broadcast, prize_pool, previous_season_name } = season;
              return (
                <TableRow className="seasons--row" key={season_id} height={60}>
                  <TableCell>{product_name}</TableCell>
                  <TableCell>{season_name}</TableCell>
                  <TableCell>{pledged_hours_broadcast == null ? '' : pledged_hours_broadcast.toLocaleString()}</TableCell>
                  <TableCell>{prize_pool === '' || prize_pool == null ? '' : '$' + prize_pool.toLocaleString()}</TableCell>
                  <TableCell>
                    {previous_season_name !== '' && previous_season_name !== null ? product_name + ' ' + previous_season_name : ''}
                    <Layout position={Position.Absolute} margin={{ x: 0.5, y: 2 }} attachRight attachTop attachBottom>
                      <MoreActions editAction={editSeason ? () => editSeason(season) : undefined} deleteAction={deleteSeason} />
                    </Layout>
                  </TableCell>
                </TableRow>
              );
            })}
        </TableBody>
      </Table>
    </StyledLayout>
  );
}

export default SeasonsList;
