import React, { useState } from 'react';
import { useHistory } from 'react-router-dom';
import { Layout } from 'twitch-core-ui';
import ErrorAlert from './error-alert';
import NoResults from './no-results';
import PaginationWrapper from './pagination-wrapper';
import SeasonsHeader from './seasons-header';
import SeasonsList from './seasons-list';
import SeasonsPlaceholder from './seasons-placeholder';
import MetricsConsole from './metrics-console';
import { ATLAS_BASE_URL } from '../utils/config';
import { useFetchPOST } from '../hooks/useFetch';
import { useLocalStorage } from '@rehooks/local-storage';
import useUpdateEffect from '../hooks/useUpdateEffect';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function Seasons() {
  const history = useHistory();
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  const searchParams = new URLSearchParams(history.location.search);
  const [currentPage, setCurrentPage] = useState(parseInt(searchParams.get('page')) || 1);
  const [filters, setFilters] = useState([
    { 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator },
    { 'filter-type': 7, key: 'end_time', v1: new Date().toISOString() }
  ]);
  const [isLegacyVisible, setIsLegacyVisible] = useState(false);

  useUpdateEffect(() => {
    setFilters([{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }]);
  }, [selectedContentCreator]);

  useUpdateEffect(() => {
    if (isLegacyVisible) {
      setFilters([{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }]);
    } else {
      setFilters([
        { 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator },
        { 'filter-type': 7, key: 'end_time', v1: new Date().toISOString() }
      ]);
    } /* eslint-disable react-hooks/exhaustive-deps */
  }, [isLegacyVisible]);

  let options = {
    limit: 100,
    page: 0,
    filters: filters,
    eventtype: 1,
    'req-cols': ['*'],
    sortdata: [
      { col: 'product_name', asc: true, ord: 0 },
      { col: 'start_time', asc: true, ord: 1 }
    ]
  };
  const [data, loading, error, fetchData] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-season-list`, options);
  const seasons = data == null || data.data == null ? [] : data.data.items;
  const totalPages = data == null || data.data == null ? 0 : data.data.total_pages;
  const metrics = data == null || data.metrics == null ? [] : data.metrics;

  useUpdateEffect(() => {
    setCurrentPage(currentPage);
    history.push({
      search: currentPage > 1 ? `?page=${currentPage}` : ''
    });
    fetchData(); /* eslint-disable react-hooks/exhaustive-deps */
  }, [currentPage, history, filters]);

  const editSeason = season => {
    history.push(`/seasons/edit/${season.season_id}`);
  };

  return (
    <>
      <Layout className="seasons">
        <SeasonsHeader isLegacyVisible={isLegacyVisible} setIsLegacyVisible={setIsLegacyVisible} />
        {error && <ErrorAlert error={error} />}
        {loading ? (
          <SeasonsPlaceholder />
        ) : seasons && seasons.length === 0 ? (
          <NoResults results="Seasons" />
        ) : (
          <>
            <SeasonsList seasons={seasons} editSeason={editSeason} />
            {totalPages > 1 && <PaginationWrapper currentPage={currentPage} totalPages={totalPages} setCurrentPage={setCurrentPage} />}
          </>
        )}
      </Layout>
      <MetricsConsole metrics={metrics} loading={loading} name="Seasons" />
    </>
  );
}

export default Seasons;
