import React, { useState } from 'react';
import Cookies from 'js-cookie';
import {
  Layout,
  StyledLayout,
  CoreText,
  TextType,
  Display,
  AlignItems,
  JustifyContent,
  SVG,
  SVGAsset,
  Tabs,
  TabItemPublicProps,
  FormGroup,
  FormGroupOrientation,
  Select,
  Input,
  InputType,
  TextArea,
  Background,
  Tag,
  FlexWrap,
  BorderRadius,
  Overflow,
  InFeatureNotification,
  NotificationType,
  Button,
  ButtonType,
  Transition,
  TransitionType,
  Position
} from 'twitch-core-ui';
import { useSelectedPermissions } from '../context/permissions-context';
import { useMasquerade } from '../context/masquerade-context';
import { utf8Convert } from '../utils/helpers';
import './styles/Settings.scss';
import NoResults from './no-results';

function Settings() {
  const permissions = useSelectedPermissions();
  const canMasquerade = permissions.includes('Can Masquerade Ldap') || permissions.includes('Can Masquerade ApiKey');
  const [activeTab, setActiveTab] = useState(0);
  const [isMasqueradeActive, setIsMasqueradeActive] = useMasquerade();
  const [masqueradeType, setMasqueradeType] = useState(localStorage.getItem('masqueradeType') || 'LDAP');
  const [ldap, setLdap] = useState(localStorage.getItem('masqueradeLdap') || '');
  const [apiKey, setApiKey] = useState(localStorage.getItem('masqueradeApiKey') || '');
  const [isMasqueradeNotificationDisplayed, setIsMasqueradeNotificationDisplayed] = useState(false);
  const [masqueradeNotificationType, setMasqueradeNotificationType] = useState(null);
  const [masqueradeNotification, setMasqueradeNotification] = useState('');
  const localStorageItems = { ...localStorage };

  const settingsTabs: TabItemPublicProps[] = [
    { label: 'Cache', onClick: () => setActiveTab(0) },
    { label: 'Masquerade', onClick: () => setActiveTab(1) }
  ];

  const removeMasquerade = () => {
    localStorage.removeItem('masqueradeType');
    localStorage.removeItem('masqueradeLdap');
    localStorage.removeItem('masqueradeApiKey');
    setIsMasqueradeActive(false);
    setLdap('');
    setApiKey('');
  };

  const clearMasquerade = () => {
    removeMasquerade();

    setMasqueradeNotification(`Masquerade has been disabled.`);
    setMasqueradeNotificationType(NotificationType.Error);
    setIsMasqueradeNotificationDisplayed(true);
  };

  const saveMasquerade = () => {
    setIsMasqueradeActive(true);
    setMasqueradeNotification(`Success! Masquerading with ${masqueradeType}: ${ldap || apiKey}.`);
    setMasqueradeNotificationType(NotificationType.Success);
    setIsMasqueradeNotificationDisplayed(true);

    localStorage.setItem('masqueradeType', masqueradeType);
    if (!!ldap) localStorage.setItem('masqueradeLdap', ldap);
    if (!!apiKey) localStorage.setItem('masqueradeApiKey', apiKey);
  };

  const onChangeMasqueradeType = e => {
    removeMasquerade();

    if (localStorageItems.masqueradeLdap || localStorageItems.masqueradeApiKey) {
      setMasqueradeNotification('Changing the masquerade type has cleared your masquerade settings. Save changes to enable masquerade.');
      setMasqueradeNotificationType(NotificationType.Info);
      setIsMasqueradeNotificationDisplayed(true);
    }
    setMasqueradeType(e.currentTarget.value);
  };

  const renderSettingsPage = () => {
    switch (activeTab) {
      case 0:
        return (
          <>
            <Layout margin={{ bottom: 2 }}>
              <CoreText type={TextType.H4}>Cookies</CoreText>
            </Layout>
            <StyledLayout margin={{ bottom: 4 }} border borderRadius={BorderRadius.Medium} overflow={Overflow.Hidden}>
              <StyledLayout background={Background.Base} padding={2} borderBottom>
                <FormGroup label="Username" orientation={FormGroupOrientation.Horizontal}>
                  <Input type={InputType.Text} value={Cookies.get('profile')} disabled />
                </FormGroup>
              </StyledLayout>
              <StyledLayout background={Background.Base} padding={2} borderBottom>
                <FormGroup label="Expire Date" orientation={FormGroupOrientation.Horizontal}>
                  <Input type={InputType.Text} value={Cookies.get('expireDate')} disabled />
                </FormGroup>
              </StyledLayout>
              <StyledLayout background={Background.Base} padding={2} borderBottom>
                <FormGroup label="Token" orientation={FormGroupOrientation.Horizontal}>
                  <TextArea value={Cookies.get('token')} rows={5} disabled />
                </FormGroup>
              </StyledLayout>
              <StyledLayout background={Background.Base} padding={2}>
                <FormGroup label="Groups" orientation={FormGroupOrientation.Horizontal}>
                  <Layout display={Display.Flex} flexWrap={FlexWrap.Wrap}>
                    {utf8Convert(Cookies.get('groups')).map(group => (
                      <Layout key={group} padding={{ bottom: 0.5, right: 1 }}>
                        <Tag label={group} disabledInteraction />
                      </Layout>
                    ))}
                  </Layout>
                </FormGroup>
              </StyledLayout>
            </StyledLayout>
            <Layout margin={{ bottom: 2 }}>
              <CoreText type={TextType.H4}>Local Storage</CoreText>
            </Layout>
            <StyledLayout margin={{ bottom: 4 }} border borderRadius={BorderRadius.Medium} overflow={Overflow.Hidden}>
              {Object.keys(localStorageItems).length === 0 ? (
                <NoResults results="localStorage items" />
              ) : (
                Object.keys(localStorageItems).map(localStorageKey => (
                  <StyledLayout key={localStorageKey} background={Background.Base} padding={2} borderBottom>
                    <FormGroup label={localStorageKey} orientation={FormGroupOrientation.Horizontal}>
                      <Input type={InputType.Text} value={localStorageItems[localStorageKey]} disabled />
                    </FormGroup>
                  </StyledLayout>
                ))
              )}
            </StyledLayout>
          </>
        );

      case 1:
        return (
          <>
            <Layout margin={{ bottom: 2 }}>
              <CoreText type={TextType.H4}>Masquerade</CoreText>
            </Layout>
            {canMasquerade ? (
              <>
                <StyledLayout margin={{ bottom: 4 }} border borderRadius={BorderRadius.Medium} overflow={Overflow.Hidden}>
                  <StyledLayout background={Background.Base} padding={2} borderBottom>
                    <FormGroup label="Type" orientation={FormGroupOrientation.Horizontal}>
                      <Select value={masqueradeType} onChange={onChangeMasqueradeType}>
                        {['LDAP', 'API Key'].map(type => (
                          <option value={type} key={type}>
                            {type}
                          </option>
                        ))}
                      </Select>
                    </FormGroup>
                  </StyledLayout>
                  <StyledLayout background={Background.Base} padding={2} borderBottom>
                    {masqueradeType === 'LDAP' && (
                      <FormGroup id="ldap" label="LDAP" hint="Twitch directory services authentication" orientation={FormGroupOrientation.Horizontal}>
                        <Input id="ldap" type={InputType.Text} value={ldap} onChange={e => setLdap(e.currentTarget.value)} />
                      </FormGroup>
                    )}
                    {masqueradeType === 'API Key' && (
                      <FormGroup id="apiKey" label="API Key" hint="Resonance API Key" orientation={FormGroupOrientation.Horizontal}>
                        <Input id="apiKey" type={InputType.Text} value={apiKey} onChange={e => setApiKey(e.currentTarget.value)} />
                      </FormGroup>
                    )}
                  </StyledLayout>
                  <StyledLayout background={Background.Base} padding={2}>
                    <FormGroup
                      label="Is Masquerade Active"
                      hint={isMasqueradeActive ? `Clear ${masqueradeType} Masquerade Settings` : ''}
                      orientation={FormGroupOrientation.Horizontal}>
                      {isMasqueradeActive ? (
                        <Layout margin={{ bottom: 0.5 }}>
                          <Button type={ButtonType.Secondary} icon={SVGAsset.Trash} onClick={clearMasquerade}>
                            Clear {masqueradeType}
                          </Button>
                        </Layout>
                      ) : (
                        <CoreText>Save Changes to enable masquerade with {masqueradeType}</CoreText>
                      )}
                    </FormGroup>
                  </StyledLayout>
                </StyledLayout>
                <Layout display={Display.Flex} justifyContent={JustifyContent.End}>
                  <Button
                    onClick={saveMasquerade}
                    disabled={!(!!ldap || !!apiKey) || localStorage.getItem('masqueradeLdap') === ldap || localStorage.getItem('masqueradeApiKey') === apiKey}>
                    Save Changes
                  </Button>
                </Layout>
              </>
            ) : (
              <InFeatureNotification
                type={NotificationType.Warning}
                message={{
                  title: 'Masquerade',
                  description: 'You do not have permissions to masquerade as another user.'
                }}
              />
            )}
          </>
        );
    }
  };

  return (
    <Layout className="settings" margin={{ bottom: 5 }} position={Position.Relative}>
      {isMasqueradeNotificationDisplayed ? (
        <StyledLayout position={Position.Absolute} attachTop attachRight elevation={1} borderRadius={BorderRadius.Medium}>
          <Transition type={TransitionType.SlideOverTop} show={isMasqueradeNotificationDisplayed}>
            <InFeatureNotification
              type={masqueradeNotificationType}
              message={{
                title: 'Masquerade',
                description: masqueradeNotification
              }}
              closeButton={{
                'aria-label': 'Close Modal',
                onClick: () => setIsMasqueradeNotificationDisplayed(false)
              }}
            />
          </Transition>
        </StyledLayout>
      ) : null}

      <Layout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between} margin={{ bottom: 2 }}>
        <Layout display={Display.Flex} alignItems={AlignItems.Center}>
          <SVG asset={SVGAsset.NavSettings} />
          <Layout margin={{ left: 1 }}>
            <CoreText type={TextType.H3}>Settings</CoreText>
          </Layout>
        </Layout>
      </Layout>
      <Layout>
        <Layout margin={{ bottom: 2 }}>
          <Tabs activeTabIndex={activeTab} tabs={settingsTabs} justifyContent={JustifyContent.Start} borderBottom={false} />
        </Layout>
        <Layout className="settings--section">{renderSettingsPage()}</Layout>
      </Layout>
    </Layout>
  );
}

export default Settings;
