import React from 'react';
import SimpleBarReact from 'simplebar-react';
import {
  Background,
  Display,
  FontSize,
  Layout,
  Position,
  StyledLayout,
  VerticalNavigationStateProvider,
  VerticalNavigation,
  VerticalNavigationGroup,
  VerticalNavigationSubHeader,
  ButtonIcon,
  SVGAsset,
  ZIndex,
  Transition,
  TransitionType,
  Tooltip,
  TooltipDirection
} from 'twitch-core-ui';
import ChannelCard from './channel-card';
import NoResults from './no-results';
import SidebarPlaceholder from './sidebar-placeholder';
import { useWindowDimensions } from '../hooks/useWindowDimensions';
import { NAV_HEIGHT } from '../utils/config';
import './styles/Sidebar.scss';
import 'simplebar/src/simplebar.css';

function Sidebar({ loading, groupings, isCollapsed, setIsCollapsed }) {
  const { height } = useWindowDimensions();

  return (
    <StyledLayout
      className={`sidebar--wrapper ${isCollapsed ? 'collapsed' : ''}`}
      attachLeft
      attachTop
      position={Position.Fixed}
      display={Display.Flex}
      fontSize={FontSize.Size5}
      background={Background.Alt}
      margin={{ y: 5 }}
      elevation={1}>
      <Layout position={Position.Absolute} zIndex={ZIndex.Above} attachRight attachTop padding={1}>
        <Tooltip label={isCollapsed ? 'Expand' : 'Collapse'} direction={TooltipDirection.Right}>
          <ButtonIcon icon={isCollapsed ? SVGAsset.ColSlideRight : SVGAsset.ColSlideLeft} onClick={() => setIsCollapsed(!isCollapsed)} />
        </Tooltip>
      </Layout>
      <Layout display={isCollapsed ? Display.Hide : Display.Block} margin={{ bottom: 5 }}>
        <Transition type={isCollapsed ? TransitionType.SlideOverRight : TransitionType.SlideOverLeft} show={!isCollapsed}>
          <div style={{ width: 240 }}>
            <SimpleBarReact style={{ maxHeight: height - NAV_HEIGHT }}>
              <VerticalNavigationStateProvider>
                {({ openGroupIDs, onCloseGroup, onOpenGroup }) => (
                  <VerticalNavigation>
                    <VerticalNavigationSubHeader>Products &amp; Channels </VerticalNavigationSubHeader>
                    {loading ? (
                      <SidebarPlaceholder />
                    ) : groupings && groupings.length === 0 ? (
                      <NoResults results="Channels" />
                    ) : (
                      groupings &&
                      groupings.map((productGroup, i) => {
                        const { product_name } = productGroup;
                        const channels = productGroup.channels;
                        channels.sort(channelSortComparator);
                        return (
                          <VerticalNavigationGroup
                            key={i}
                            label={product_name}
                            open={openGroupIDs.includes(product_name)}
                            onOpen={() => onOpenGroup(product_name)}
                            onClose={() => onCloseGroup(product_name)}>
                            {channels.map((channel, j) => {
                              const { channel_login, channel_status, viewers, channel_category, channel_profile_image } = channel;
                              return (
                                <Layout padding={{ y: 0.5 }} key={j}>
                                  <ChannelCard
                                    login={channel_login}
                                    status={channel_status}
                                    viewers={viewers}
                                    category={channel_category}
                                    profileImage={channel_profile_image}
                                  />
                                </Layout>
                              );
                            })}
                          </VerticalNavigationGroup>
                        );
                      })
                    )}
                  </VerticalNavigation>
                )}
              </VerticalNavigationStateProvider>
            </SimpleBarReact>
          </div>
        </Transition>
      </Layout>
    </StyledLayout>
  );
}

function channelSortComparator(a, b) {
  const onlineStatus = "Online";
  if (a.channel_status !== b.channel_status) {
    // online status gets priority. Otherwise fall back to other stuff
    if (a.channel_status === onlineStatus) {
      return -1;
    } else if (b.channel_status === onlineStatus) {
      return 1;
    }
  }
  
  if (a.channel_status === b.channel_status && a.channel_status === onlineStatus) {
    // If both are online, sort by viewers
    return a.viewers > b.veiwers ? -1 : 1;
  }

  // otherwise sort alphabetically
  return (a.channel_login < b.channel_login) ? -1 : 1;
}

export default Sidebar;
