import useLocalStorage from '@rehooks/local-storage';
import React, { createContext, useContext, useEffect, useState } from 'react';
import { useFetchGET, useFetchPOST } from '../hooks/useFetch';
import useUpdateEffect from '../hooks/useUpdateEffect';
import { ATLAS_BASE_URL } from '../utils/config';
import { Product, TopicOptions } from '../utils/types';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';
export interface DataContextValue {
    allTopicsFormatsOptions?: TopicOptions[];
    productsList?: Product[];
    loading: boolean;
}

const DataContext = createContext<DataContextValue>({
    loading: true,
});

export default DataContext;

function useDataContext() {
  const context = useContext(DataContext);
  if (!context) {
    throw new Error(`Error with useDataContext`);
  }
  return context;
}

function getProductsOptions(selectedContentCreator: string) {
    return {
        eventtype: 1,
        limit: 1000,
        'req-cols': ["*"],
        filters: [{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }],
        sortdata: [{ ord: 0, col: 'product_name', asc: true }]
    }
}

function DataContextProvider(props) {
    const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
    const [topicsData, topicsLoading, topicsError] = useFetchGET(`${ATLAS_BASE_URL}/event/get-topics-formats`);
    const [productsData, productsLoading, productsError, fetchProductsList] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-product-list`, getProductsOptions(selectedContentCreator));
    const [topicOptions, setTopicOptions] = useState<TopicOptions[] | undefined>()
    const [productsList, setProductsList] = useState<Product[] | undefined>();

    useUpdateEffect(() => {
        const searchOptions = getProductsOptions(selectedContentCreator);
        fetchProductsList(searchOptions);
    }, [selectedContentCreator]);

    useEffect(() => {
        if (!topicsLoading) {
            const data = topicsData?.data || [];
            setTopicOptions([
                ...data
            ]);
        }

        if (topicsError) {
            console.error("Error fetching topics data", topicsError)
        }
    }, [topicsLoading, topicsData, topicsError]);

    useEffect(() => {
        if (!productsLoading) {
            const data = productsData?.data?.items || [];
            setProductsList(data);
        }

        if (productsError) {
            console.error("Error fetching products data", productsError)
        }
    }, [productsData, productsLoading, productsError]);

    const loading = topicsLoading || productsLoading;
    return <DataContext.Provider value={{
        allTopicsFormatsOptions: topicOptions,
        productsList,
        loading,
    }} {...props} />;
}

export { DataContextProvider, useDataContext };
