import React, { createContext, useContext, useState, useMemo } from 'react';
import { AtlasEvent, WorkspaceEvent } from '../utils/types';
import { v4 as uuidv4 } from 'uuid';

const EventsWorkspaceContext = createContext<EventsWorkspaceProviderContextProps>({
  eventsWorkspaceList: [],
  setEventsWorkspaceList: () => null,
  addEventsToWorkspaceList: () => null,
});

function useEventsWorkspaceList() {
  const context = useContext(EventsWorkspaceContext);
  if (!context) {
    throw new Error(`useEventsWorkspaceList must be used within a EventsWorkspaceProvider`);
  }
  return context;
}

interface EventsWorkspaceProviderContextProps {
  eventsWorkspaceList: WorkspaceEvent[],
  setEventsWorkspaceList: React.Dispatch<React.SetStateAction<WorkspaceEvent[]>>,
  addEventsToWorkspaceList: (events: AtlasEvent[]) => void,
}

function EventsWorkspaceProvider(props) {
  const [eventsWorkspaceList, setEventsWorkspaceList] = useState<WorkspaceEvent[]>(JSON.parse(localStorage.getItem('eventsWorkspaceList')) || []);
  
  const value = useMemo<EventsWorkspaceProviderContextProps>(() => { 
    const addEventsToWorkspaceList = (events: AtlasEvent[]) => {
      setEventsWorkspaceList((prevEventsWorkspaceList) => {
        const newEvents = [...prevEventsWorkspaceList];
        for (const event of events) {
          newEvents.push({
            workspace_id: uuidv4(),
            ...event,
          });
        }
        return newEvents;
      });
    }

    return { eventsWorkspaceList, setEventsWorkspaceList, addEventsToWorkspaceList }
  }, [eventsWorkspaceList]);
  return <EventsWorkspaceContext.Provider value={value} {...props} />;
}

export { EventsWorkspaceProvider, useEventsWorkspaceList };
