import { ApolloQueryResult } from "@apollo/client";
import { loader } from "graphql.macro";
import React, { createContext, useContext } from "react";
import { apolloClient } from "../utils/apolloClient";
import { GetChannelsResults, GetUsersQueryResult, SearchCategoriesQueryResult, SearchCategoriesResults } from "../utils/gqlTypes";

const GetUsers = loader('../gql/getUsers.gql');
const SearchCategories = loader('../gql/searchCategories.gql');


export type GetChannelsCallback = (channelNames: string[]) => Promise<GetChannelsResults | null>;
export type SearchCategoriesCallback = (query: string) => Promise<SearchCategoriesResults | null>;

export interface GQLContextValue {
    getChannels: GetChannelsCallback;
    searchCategories: SearchCategoriesCallback;
}

const GQLContext = createContext<GQLContextValue>({
    getChannels: () => null,
    searchCategories: () => null,
});

export function useGQLContext() {
    const context = useContext(GQLContext);
    if (!context) {
      throw new Error(`Error with useGQLContext`);
    }

    return context;
}

export function GQLContextProvider(props) {
    const searchCategories = React.useCallback(async (query: string): Promise<SearchCategoriesResults | null> => {
        const result = await apolloClient.query<SearchCategoriesQueryResult | null>({
            query: SearchCategories,
            variables: {
                query,
            }
        });

        if (isError(result)) {
            return null;
        }

        if (result?.data?.searchCategories?.edges) {
            return {
                categories: result.data.searchCategories.edges.map(edge => edge.node),
            };
        };

        return null;
    }, []);

    const getChannels = React.useCallback(async (channelNames: string[]): Promise<GetChannelsResults | null> => {
        const result = await apolloClient.query<GetUsersQueryResult | null>({
            query: GetUsers,
            variables: {
                logins: channelNames,
            }
        });
        
        if (isError(result)) {
            return null;
        }

        if (!result.data.users) {
            console.error("Failed to fetch channels: GQL error");
            return null;
        }

        const channelResults: GetChannelsResults = {
            results: [],
            notFound: [],
        };

        const found = new Set<string>();
        for (const item of result.data.users) {
            if (item?.id && item?.login) {
                channelResults.results.push(item);
                found.add(item.login);
            }
        }

        for (const login of channelNames) {
            if (!found.has(login)) {
                channelResults.notFound.push(login);
            }
        }

        return channelResults;
    }, []);

    return <GQLContext.Provider value={{
        searchCategories,
        getChannels,
    }} {...props} />;
}

function isError<T>(result: ApolloQueryResult<T>) {
    if (result.error || result.errors) {
        console.error("A GQL error has ocurred", {
            error: result.error, 
            errors: result.errors,
        });

        return true;
    }

    return false;
}