import { useState, useEffect } from 'react';
import Cookies from 'js-cookie';
import { fetchWithTimeout } from '../utils/helpers';
import { useMasquerade } from '../context/masquerade-context';

function useFetchGET(url) {
  const [isMasqueradeActive] = useMasquerade();
  const masqueradeType = localStorage.getItem('masqueradeType');
  const masqueradeLdap = localStorage.getItem('masqueradeLdap');
  const masqueradeApiKey = localStorage.getItem('masqueradeApiKey');
  const isMasqueradeValid = isMasqueradeActive && (masqueradeLdap !== null || masqueradeApiKey !== null);

  const options = {
    method: 'GET',
    headers: {
      Accept: 'application/json',
      'Content-Type': 'application/json',
      Authorization: `Bearer ${Cookies.get('token')}`,
      Masquerade: isMasqueradeValid ? (masqueradeType === 'LDAP' ? `Bearer ${masqueradeLdap}` : `ApiKey ${masqueradeApiKey}`) : ''
    }
  };

  const [data, setData] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  async function fetchData() {
    setLoading(true);

    try {
      const response = await fetchWithTimeout(url, options);
      const json = await response.json();
      console.log(`${url.split('/event')[1]}:`, json);
      if (json.success) {
        setData(json);
        setLoading(false);
      } else {
        switch (json.data['status-code']) {
          case 401:
            setError(`401 (Unauthorized) for ${url}`);
            setLoading(false);
            break;
          case 404:
            setError(`404 for ${url}`);
            setLoading(false);
            break;
          case 400:
            setError(`400 (Bad Request) for ${url}`);
            setLoading(false);
            break;
          default:
            setError(`Atlas backend not found - JSON request for ${url} not successful`);
            setLoading(false);
            break;
        }
      }
    } catch (err) {
      setError(err);
      setLoading(false);
      console.error(`Error with url ${url}`, err);
    }
  }

  useEffect(() => {
    fetchData(); /* eslint-disable react-hooks/exhaustive-deps */
  }, []);

  return [data, loading, error, fetchData];
}

function useFetchPOST(url: string, body = {}) {
  const [isMasqueradeActive] = useMasquerade();
  const masqueradeType = localStorage.getItem('masqueradeType');
  const masqueradeLdap = localStorage.getItem('masqueradeLdap');
  const masqueradeApiKey = localStorage.getItem('masqueradeApiKey');
  const isMasqueradeValid = isMasqueradeActive && (masqueradeLdap !== null || masqueradeApiKey !== null);

  const [data, setData] = useState([]);
  const [loading, setLoading] = useState(true);
  const options = {
    method: 'POST',
    headers: {
      Accept: 'application/json',
      'Content-Type': 'application/json',
      Authorization: `Bearer ${Cookies.get('token')}`,
      Masquerade: isMasqueradeValid ? (masqueradeType === 'LDAP' ? `Bearer ${masqueradeLdap}` : `ApiKey ${masqueradeApiKey}`) : ''
    },
    body: JSON.stringify(body)
  };

  const [error, setError] = useState(null);

  async function fetchData(body?: any) {
    setLoading(true);

    const opts = body ? {
      ...options,
      body: JSON.stringify(body),
    } : options;

    try {
      const response = await fetchWithTimeout(url, opts);
      const json = await response.json();
      console.log(`${url.split('/event')[1]}:`, json);
      if (json.success) {
        setData(json);
        setLoading(false);
      } else {
        const code = json?.data ? json.data['status-code'] : null;
        switch (code) {
          case 400:
            setError(`400 (Bad Request) for ${url}`);
            setLoading(false);
            break;
          case 401:
            setError(`401 (Unauthorized) for ${url}`);
            setLoading(false);            break;
          case 403:
            setError(`403 Access Denied/Forbidden for ${url}`);
            setLoading(false);
            break;
          case 404:
            setError(`404 for ${url}`);
            setLoading(false);
            break;
          default:
            setError(`Atlas backend not found - JSON request for ${url} not successful`);
            setLoading(false);
            break;
        }
      }
    } catch (err) {
      setError(`${err} - JSON request for ${url} not successful`);
      setLoading(false);
      console.error(`Error with url ${url}`, err);
    }
  }

  useEffect(() => {
    fetchData(); /* eslint-disable react-hooks/exhaustive-deps */
  }, []);

  return [data, loading, error, fetchData];
}

export { useFetchPOST, useFetchGET };
