import { parseCSV } from "./csvParse";

const csvString = 
`thing,blep
abc,123
def,567`

const csvStringEscaped = 
`thi""ng,"blep,aaa"
"abc,and more",123
d""ef,"567,555"
"de""f","1111"`

const badCsvString1 = 
`thing,blep
abc,123,
def,567`


const badCsvString2 = 
`thing,blep
abc,123,
def`

const normalHeaders = ["thing","blep","urg"];

describe("parseCSV", () => {
    it("Parses CSV", () => {
        const results = parseCSV(csvString, {knownHeaders: normalHeaders });
        expect(results[0].thing).toBe("abc");
        expect(results[0].blep).toBe("123");
        expect(results[1].thing).toBe("def");
        expect(results[1].blep).toBe("567");
    });

    it("Applies special rules", () => {
        const results = parseCSV(csvString, { knownHeaders: normalHeaders, rules: {
            "blep": (value) => {
                let num = +value;
                num = num + 1;
                return num;
            }   
        }});
        expect(results[0].thing).toBe("abc");
        expect(results[0].blep).toBe(124);
        expect(results[1].thing).toBe("def");
        expect(results[1].blep).toBe(568);
    });

    it("Properly parses quoted CSV", () => {
        const header1 = `thi"ng`;
        const header2 = `blep,aaa`;
        const results = parseCSV(csvStringEscaped, {knownHeaders: [header1, header2, "urg"] });
        expect(results[0][header1]).toBe("abc,and more");
        expect(results[0][header2]).toBe("123");
        expect(results[1][header1]).toBe(`def`);
        expect(results[1][header2]).toBe("567,555");
        expect(results[2][header1]).toBe(`de"f`);
        expect(results[2][header2]).toBe("1111");
    });

    it("Properly errors on unknown headers", () => {
        try {
            parseCSV(csvString, {knownHeaders: ["loluwontfindme","samehere", "thing"] });
        } catch (err) {
            return;
        }

        fail("Should catch error");
    });

    it("Properly errors on mismatched item count (too many)", () => {
        try {
            parseCSV(badCsvString1, {knownHeaders: normalHeaders });
        } catch (err) {
            return;
        }

        fail("Should catch error");
    });

    it("Properly errors on mismatched item count (too few)", () => {
        try {
            parseCSV(badCsvString2, {knownHeaders: normalHeaders });
        } catch (err) {
            return;
        }

        fail("Should catch error");
    });
});
