import { allowedEventImportHeaders, EventImportData } from "./importModels";
import { coStreamingSettingsOptions } from "./options";
import { Product, TopicOptions } from "./types";

const daysMs = 24*60*60*1000;
const fiveDaysMs =  5 * daysMs;
const tenDaysMs = 10 * daysMs;

/**
 * Gets a random item from the list
 */
function getRandom(list: string[]) {
    if (!list?.length) {
        return "";
    }

    const rand = Math.floor(Math.random() * list.length);
    return list[rand];
}

/**
 * Gets the "next" item based on the running counter, by
 * cycling thru list using modulo.
 */
function cycleThrough(counter: number, list: string[]) {
    return list[counter % list.length];
}

function randInt(min: number, max: number) {
    return Math.floor((Math.random() * (max - min)) + min);
}

const channels = ["capcomfighters;callofduty;fortnite", "flavioliravioli;ninja", "flavioliravioli"];
const games = ["Fun Game", "Game Time", "Call Of Battlewar", "Gacha Quest", "League of the Ancients"];
const distributionOptions = ["Yes", "No"];

const excludedHeaders = new Set<keyof EventImportData>(["game_name", "exclusivity"]);
const headers = allowedEventImportHeaders.filter(a => !excludedHeaders.has(a));

function removeDateEnd(dateTime: string) {
    // Removes the milliseconds and Z from the ISO format, since
    // currently our dates don't work very well with this.
    return dateTime?.replace(/(\.\d+)?Z$/, '');
}

function createRandomEvent(counter: number, productName: string, topic: string, format: string): EventImportData {
    // Start sometime between 5-10 days from now and
    // end sometime between 5-10 days from start date
    const startOffset = randInt(fiveDaysMs, tenDaysMs);
    const endOffset = randInt(fiveDaysMs, tenDaysMs);
    const startTime = new Date(new Date().getTime() + startOffset);
    const endTime = new Date(startTime.getTime() + endOffset);

    const category = getRandom(games);
    
    // for these, we want to show each example, so just cycle thru the list using modulo 
    const channel_names = cycleThrough(counter, channels);
    const costreaming_settings = cycleThrough(counter, coStreamingSettingsOptions);
    const distribution = cycleThrough(counter, distributionOptions);
    const estimated_average_ccv = randInt(10, 100000).toString();
    
    return {
        event_name: `Event ${counter}`,
        product_name: productName,
        topic,
        format,
        start_time: removeDateEnd(startTime.toISOString()),
        end_time: removeDateEnd(endTime.toISOString()),
        category,
        channel_names,
        costreaming_settings,
        distribution,
        estimated_average_ccv
    }
}

function createEventRow(data: EventImportData) {
    const values = [];
    for (const header of headers) {
        values.push(data[header] ?? "");
    }

    return values.join(",");
}

export function generateEventsCsvTemplate(productsList: Product[], allTopicsFormatsOptions: TopicOptions[]) {
    let counter = 0;
    const rows = [];
    for (const product of productsList) {
        const topic = product.topic;
        const options = allTopicsFormatsOptions.find(a => a.topic === topic);
        if (!options?.formats?.length) {
            console.warn(`No formats found for topic '${topic}' `);
            continue;
        }

        for (const format of options.formats) {
            const event = createRandomEvent(counter, product.product_name, topic, format.format);
            const row = createEventRow(event);
            rows.push(row);
            counter++;
        }
    }

    return `${headers.join(",")}\n${rows.join("\n")}`;
}
