import { timezoneOptions } from './options';
import moment from 'moment';

export function createDateString(year, month, day) {
  return year + '-' + ('0' + month).slice(-2) + '-' + ('0' + day).slice(-2);
}

declare global {
  interface Date {
    stdTimezoneOffset(): number;
    isDstObserved(): boolean;
  }
}

// eslint-disable-next-line no-extend-native
Date.prototype.stdTimezoneOffset = function() {
  var jan = new Date(this.getFullYear(), 0, 1);
  var jul = new Date(this.getFullYear(), 6, 1);
  return Math.max(jan.getTimezoneOffset(), jul.getTimezoneOffset());
};

// eslint-disable-next-line no-extend-native
Date.prototype.isDstObserved = function() {
  return this.getTimezoneOffset() < this.stdTimezoneOffset();
};

export function setMomentDateTime(dateStr) {
  // dateStr = "2020-04-15T22:56:22"
  if (dateStr == null) return;
  const tz = localStorage.getItem('timezone');
  if (!tz) {
    return moment(dateStr);
  }

  const tzObj = JSON.parse(tz);
  const isDST = new Date(dateStr).isDstObserved() && tzObj.DST;
  return moment(dateStr + '+00:00')
    .utcOffset(-1 * (isDST ? parseInt(tzObj && tzObj.offset) + 1 : tzObj && tzObj.offset)) // converting from current timezone to UTC offset
    .format('YYYY-MM-DDTHH:mm:ss');
}

export function getMomentDateTime(dateStr) {
  // dateStr = "2020-04-15T22:56:22"
  if (dateStr === null) return;
  const tz = localStorage.getItem('timezone');
  const tzObj = JSON.parse(tz);
  if (tz === null) {
    return moment(dateStr);
  }
  const isDST = new Date(dateStr).isDstObserved() && tzObj.DST;
  return moment(dateStr + '+00:00').utcOffset(isDST ? parseInt(tzObj && tzObj.offset) + 1 : tzObj && tzObj.offset);
}

export function getMomentDaysDiff(startTime, endTime) {
  if (startTime === null || endTime === null) return;
  const convertedStartTime = getMomentDateTime(startTime)?.format('D');
  const convertedEndTime = getMomentDateTime(endTime)?.format('D');
  const daysDiff = parseInt(convertedEndTime) - parseInt(convertedStartTime);
  return daysDiff;
}

export function getMomentDate(dateStr, momentFormat = 'MM/DD/YYYY') {
  // dateStr = "2020-04-15T22:56:22"
  if (dateStr === null) return;
  return getMomentDateTime(dateStr)?.format(momentFormat); // 04/15/2010
}

export function getMomentTime(dateStr) {
  // dateStr = "2020-04-15T22:56:22"
  if (dateStr === null) return;
  return getMomentDateTime(dateStr)?.format('HH:mm:ss'); // 23:59:00
}

export function getMomentTimeAmPm(dateStr) {
  // dateStr = "2020-04-15T22:56:22"
  if (dateStr === null) return;
  return getMomentDateTime(dateStr)?.format('h:mma'); // 11:59am
}

export function getMomentFullDateTime(dateStr) {
  if (dateStr === null) return;
  // dateStr = "2020-04-15T22:56:22"
  return getMomentDateTime(dateStr)?.format('M/D/YYYY, hh:mmA'); // 7/13/1988, 11:00AM
}

export function getMomentFullDate(dateStr) {
  // dateStr = "2020-04-15T22:56:22"
  if (dateStr === null) return;
  return getMomentDateTime(dateStr)?.format('dddd, MMMM D, YYYY'); // Saturday, July 13, 1988
}

export function getDefaultTimezoneOption() {
  let offset = getTimezoneOffset();
  let defaultOption = null;
  timezoneOptions.forEach(tzOption => {
    if (tzOption && tzOption.offset === offset) {
      defaultOption = tzOption;
    }
  });

  return defaultOption || { name: '(GMT -8:00) Pacific Time (US & Canada)', value: 'Pacific Standard Time', offset: '-08:00', DST: true };
}

export function getTimezoneOffset() {
  const today = new Date();
  const jan = new Date(today.getFullYear(), 0, 1);
  const jul = new Date(today.getFullYear(), 6, 1);

  const tzOffset = Math.max(jan.getTimezoneOffset(), jul.getTimezoneOffset());

  function pad(number, length) {
    let str = '' + number;
    while (str.length < length) {
      str = '0' + str;
    }
    return str;
  }

  let offset =
    (tzOffset < 0 ? '+' : '-') + // Note the reversed sign
    pad(Math.abs(tzOffset / 60), 2) +
    ':' +
    pad(Math.abs(tzOffset % 60), 2);

  return offset;
}
