import Fuse from "fuse.js"

/**
 * Given a value and a list of options, gets the best match using
 * the Fuse.JS fuzzy-matching algorithm, based on a default threshold
 * of 0.6. This should match most close matches properly, while returning
 * null if nothing in the list of options resembles value closely.
 * 
 * Example:
 * 
 *    options: ["Bob", "Bill", "Flavio"]
 * 
 * A value of "flav" will return "Flavio".
 * 
 */
export function fuzzySelect(value: string, options: string[]): string | null {
    if (value === undefined || value === null) {
        return null;
    }

    const fuse = new Fuse(options, { })
    const val = fuse.search(value);
    if (val.length) {
        return val[0].item;
    }

    // If original fuzzy matching doesn't work, try doing a more naive
    // compare using substrings (case insensitive, with spaces and punctuation removed)
    const valueSimplified = simplify(value);
    const optionsSimplified = options.map(a => simplify(a.toLowerCase()));
    for (let i = 0; i < options.length; i++) {
        const option = optionsSimplified[i];
        if (option.includes(valueSimplified) || valueSimplified.includes(option)) {
            return options[i];
        }
    }

    return null;
}

/**
 * Removes spaces and punctuation from the string, and makes it lowercase
 */
export function simplify(item: string) {
    const itemLower = item.toLowerCase();
    return itemLower.replace(/[\s_;!@#$%^&*()-]/g, "");
}

/**
 * Given value and set of objects (options), as well as a list of property names (keys),
 * performs a shallow search on the objects using the provided keys in order to find the
 * best possible valid match.
 * 
 * Example:
 * 
 *   options: [
 *     {
 *       "title": "Old Man's War",
 *       "author": "John Scalzi",
 *       "tags": ["fiction"]
 *     },
 *     {
 *       "title": "The Lock Artist",
 *       "author": "Steve",
 *       "tags": ["thriller"]
 *     }
 *   ]
 *   
 *   keys: ["tags", "author"]
 * 
 * A value of "fiction" will return the first object, and a value of "Steve" will
 * return the second object.
 */
export function fuzzySearchSelect<T>(value: string, options: T[], keys: string[]): T | null {
    if (value === undefined || value === null) {
        return null;
    }

    const fuse = new Fuse(options, { keys })
    const val = fuse.search(value);
    return val.length ? val[0].item : null;
}