import { saveAs } from 'file-saver';

export function getPercentage(num) {
  return (num * 100).toFixed(1) + '%';
}

export function hyphenify(text = '') {
  return text
    .toString()
    .toLowerCase()
    .replace(/\s/g, '-');
}

export function abbreviateNumber(num) {
  if (num === null || num === undefined || num === '') return;
  return num > 999999999
    ? (num / 1000000000).toFixed(1) + 'b'
    : num > 999999
    ? (num / 1000000).toFixed(1) + 'm'
    : num > 999
    ? (num / 1000).toFixed(1) + 'k'
    : Math.abs(parseInt(num));
}

export function camelCaseToUnderscore(key) {
  return key.replace(/([A-Z])/g, '_$1').toLowerCase();
}

export function buildEventFilters(searchParams, selectedContentCreator) {
  if (searchParams === null) return;
  const filterArr = [];
  for (let param of searchParams) {
    if (param[0] === 'page' || param[1] === '') {
      continue;
    }
    if (param[1].length) {
      let filterObj;

      switch (param[0]) {
        case 'eventName':
          // <InputText />
          filterObj = { 'filter-type': 4, key: camelCaseToUnderscore(param[0]), v1: param[1] };
          break;

        case 'products':
          // <Multiselect />
          // { 'filter-type': 5, key: 'usertype', a1: ['Partner', 'Affiliate'] }
          filterObj = { 'filter-type': 5, key: 'product_id', a1: param[1].split(',') };
          break;

        case 'topic':
        case 'gameName':
        case 'channels':
        case 'seasonName':
        case 'subtopic':
        case 'format':
        case 'subformat':
        case 'twitchInvolvement':
        case 'costreamingSettings':
          // <Multiselect />
          // { 'filter-type': 5, key: 'usertype', a1: ['Partner', 'Affiliate'] }
          filterObj = { 'filter-type': 5, key: camelCaseToUnderscore(param[0]), a1: param[1].split(',') };
          break;

        case 'distribution':
          // <Checkbox />
          // { 'filter-type': 1, key: 'ispremium', v1: false }
          filterObj = { 'filter-type': 1, key: camelCaseToUnderscore(param[0]), v1: typeof param[1] === 'string' ? param[1] === 'on' : param[1][0] };
          break;

        case 'timeline':
          // <DateRange />
          // { 'filter-type': 2, key: 'gamecount', 'is-inclusive': true, v1: 1, v2: 999999 }
          filterObj = { 'filter-type': 2, key: 'start_time', v1: param[1].split(',')[0], v2: param[1].split(',')[1] };
          break;
      }
      filterArr.push(filterObj);
    }
  }
  if (selectedContentCreator !== '') {
    filterArr.push({ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator });
  }
  return filterArr;
}

export function getTopics(data) {
  if (data == null) return;
  let topicOptions = [];
  topicOptions = data && data.map(option => ({ topic: option.topic, topic_id: option.topic_id }));
  return topicOptions;
}

export function getSubtopics(selectedTopics, data) {
  if (data == null || selectedTopics == null) return;
  let subtopicOptions = [];

  data.forEach(option => {
    for (let selectedTopic of selectedTopics) {
      if (selectedTopic.topic === option.topic) {
        subtopicOptions = [...subtopicOptions, ...option.sub_topics];
      }
    }
  });

  return Array.from(new Set(subtopicOptions));
}

export function getFormats(selectedTopics, data) {
  if (data == null || selectedTopics == null) return;
  let formatOptions = [];

  data.forEach(option => {
    for (let selectedTopic of selectedTopics) {
      if (selectedTopic.topic === option.topic) {
        option.formats.map(formatObj => formatOptions.push(formatObj.format));
      }
    }
  });

  return Array.from(new Set(formatOptions));
}

export function getSubformats(selectedTopics, selectedFormats, data) {
  if (data == null || selectedTopics == null || selectedFormats == null) return;

  let formatOptions = [];
  data.forEach(option => {
    for (let selectedTopic of selectedTopics) {
      if (selectedTopic.topic === option.topic) {
        option.formats.map(formatObj => formatOptions.push(formatObj));
      }
    }
  });

  let subformatOptions = [];
  formatOptions.forEach(formatObj => {
    for (let selectedFormat of selectedFormats) {
      if (selectedFormat === formatObj.format) {
        subformatOptions = [...subformatOptions, ...formatObj.sub_formats];
      }
    }
  });

  return Array.from(new Set(subformatOptions));
}

export function isObjectEmpty(obj) {
  for (var x in obj) {
    return false;
  }
  return true;
}

export function textify(text) {
  return text.toString().replace(/[ -_]+/g, ' ');
}

export const fetchWithTimeout = (url: string, options = {} as any, time = 10000) => {
  // Ensure Masquerade is in use for all requests
  const masqueradeType = localStorage.getItem('masqueradeType');
  const masqueradeApiKey = localStorage.getItem('masqueradeApiKey');
  const masqueradeLdap = localStorage.getItem('masqueradeLdap');
  const isMasqueradeValid = !!masqueradeType;
  if (isMasqueradeValid) {
    const masquerade = isMasqueradeValid ? (masqueradeType === 'LDAP' ? `Bearer ${masqueradeLdap}` : `ApiKey ${masqueradeApiKey}`) : '' ;
    const headers = options.headers ?? {};
    options.headers = { ...headers, Masquerade: masquerade };
  }

  const controller = new AbortController();
  const config = { ...options, signal: controller.signal };
  const isLocal = window.location.host.includes('localhost');

  setTimeout(
    () => {
      controller.abort();
    },
    isLocal ? 60000 : time
  );

  return fetch(url, config)
    .then(response => {
      if (!response.ok) {
        throw new Error(`${response.status}: ${response.statusText}`);
      }
      return response;
    })
    .catch(err => {
      if (err.name === 'AbortError') {
        throw new Error('Response timed out');
      }
      throw new Error(err.message);
    });
};

export const utf8Convert = cookieStr => {
  const convertedStr = cookieStr.replace('%22', '"').replace('%2c', ',');
  return JSON.parse(convertedStr);
};

export function downloadTextFile(text: string, filename: string) {
  var blob = new Blob([text], {type: "text/plain;charset=utf-8"});
  saveAs(blob, filename);
}
