import { GetChannelsCallback } from "../context/gql-context";
import { fuzzySearchSelect, fuzzySelect } from "./fuzzySearch";
import { EventImportData } from "./importModels";
import { channelTypeOptions, coStreamingSettingsOptions } from "./options";
import { AtlasEvent, Product, TopicOptions } from "./types";

export interface EventParsingError {
    messages: string[];
    event: AtlasEvent;
    problemFields: string[];
}

export interface EventParsingResults {
    errors: EventParsingError[];
    parsedEvents: AtlasEvent[];
}

interface ListingData {
    productsList: Product[];
    currentContentCreatorID: string;
    allTopicsFormatsOptions: TopicOptions[];
    getChannels: GetChannelsCallback;
}

interface ParseResult {
    error?: string;
    problemFields?: string[];
}

/**
 * Given the list of EventImportData and various lists such
 * as list of Products, this does fuzzy matches and validation
 * on each relevant item to ensure the events are valid, converting
 * each item into a parsed AtlasEvent.
 * 
 * Should be used for importing data values, such as after importing
 * from csv files.
 * 
 * The optional "onUpdateProgress" callback can be used to show the progress
 * update on the UI, using the progess value (0 to 100) and status string.
 * 
 * Returns list of successfully parsed events and list of errors.
 */
export async function processEventValues(
    events: EventImportData[], 
    data: ListingData, 
    onUpdateProgress?: (progress: number, status: string) => void
    ) : Promise<EventParsingResults> {
    const { productsList, currentContentCreatorID, allTopicsFormatsOptions, getChannels } = data;
    const allResults: EventParsingResults = {
        errors: [],
        parsedEvents: [],
    };

    for (let i = 0; i < events.length; i++) {
        const item = events[i];
        const progress = (i / events.length) * 100;
        onUpdateProgress?.(progress, `Processing ${item.event_name ?? ""}`);

        const parsed: Partial<AtlasEvent> = {};

        parsed.start_time = item.start_time;
        parsed.end_time = item.end_time;

        parsed.estimated_average_ccv = +item.estimated_average_ccv;
        if (isNaN(parsed.estimated_average_ccv)) {
            parsed.estimated_average_ccv = undefined;
        }

        // Both distribution and exclusivity are valid options for this field
        const distribution = item.distribution ?? item.exclusivity;
        parsed.distribution = (distribution === "true");

        // Both game_name and category are valid options for this field
        parsed.game_name = item.category ?? item.game_name;

        // deprecated fields - set a sensible default
        parsed.twitch_involvement = "";

        const results: ParseResult[] = [];

        results.push(processPCCId(parsed, currentContentCreatorID));
        results.push(processEventName(item, parsed));
        results.push(processProductName(item, parsed, productsList));
        results.push(processTopic(item, parsed, allTopicsFormatsOptions));
        results.push(processCoStreamSettings(item, parsed));
        results.push(await processChannels(item, parsed, getChannels));

        const messages = [];
        const problemFields = [];
        for (const result of results) {
            if (result.error) {
                messages.push(result.error);
            }

            if (result.problemFields?.length) {
                problemFields.push(...result.problemFields);
            }
        }

        const fullEvent = parsed as AtlasEvent;
        if (messages.length || problemFields.length) {
            allResults.errors.push({
                messages,
                event: fullEvent,
                problemFields,
            });
        } else {
            allResults.parsedEvents.push(fullEvent);
        }
    }

    onUpdateProgress?.(100, `Done!`);
    
    return allResults;
}

async function processChannels(item: EventImportData, parsed: Partial<AtlasEvent>, getChannels: GetChannelsCallback): Promise<ParseResult> {
    const problemFields = ["channels"];

    if (!item.channel_names) {
        return {
            error: "No channel specified; at least one channel required",
            problemFields
        };
    }

    // get list of channels, filtering out empty strings and such
    let channels = item.channel_names.split(";");
    channels = channels.filter(channel => !!(channel || "").trim());

    const data = await getChannels(channels);
    
    if (!data) {
        return {
            error: `Unable to verify channel list: [ ${channels.join(", ")} ]`,
            problemFields,
        };
    }

    parsed.channels = data.results.map(channel => { 
        return {
            channel_id: +channel.id,
            channel_login: channel.login,
            profile_image: channel.profileImageURL,

            // default the channel_type to "PCC Owned - Primary"
            channel_type: channelTypeOptions[0],
        };
    });

    parsed.channel_count = parsed.channels.length;

    if (data.notFound.length) {
        const failedItems = data.notFound.join(", ");
        return {
            error: `The following channels were not found: [ ${failedItems} ]`,
            problemFields,
        }
    }

    return {};
}

function processEventName(item: EventImportData, parsed: Partial<AtlasEvent>): ParseResult {
    if (!item.event_name) {
        parsed.event_name = "unknown";
        return {
            error: "event_name not specified",
            problemFields: ["event_name"],
        }
    } else {
        parsed.event_name = item.event_name;
    }

    return {};
}

function processProductName(item: EventImportData, parsed: Partial<AtlasEvent>, productsList: Product[]): ParseResult {
    // narrow down product_name through fuzzy matching
    const product = fuzzySearchSelect(item.product_name ?? "", productsList, ["product_name"]);
    if (!product) {
        const productNames = productsList.map((a) => a.product_name);
        return {
            error: `Unable to parse product_name. Received: [ ${item.product_name} ]. Valid Options: [ ${productNames.join(", ")} ]`,
            problemFields: ["product_name"],
        }
    }

    parsed.product_id = product.product_id;
    parsed.product_name = product.product_name;
    parsed.topic = product.topic;
    return {};
}

function processTopic(item: EventImportData, parsed: Partial<AtlasEvent>, allTopicsFormatsOptions: TopicOptions[]): ParseResult {
    // topic may have been parsed from product earlier, or included directly
    const topic = parsed.topic ?? item.topic;
    if (!topic) {
        return {
            error: "Unable to determine topic. Check that you specified the 'topic' or 'product_name' fields.",
            problemFields: ["topic"],
        }
    }

    const topicOptions = allTopicsFormatsOptions.find(a => a.topic === topic);
    if (topicOptions) {
        const formats = topicOptions.formats.map((a) => a.format);
        if (formats) {
            parsed.format = fuzzySelect(item.format, formats);
            if (!parsed.format) {
                return {
                    error: `Unable to determine format from [ ${item.format} ]. Valid formats for topic [ ${topic} ] are: [ ${formats.join(", ")} ]`,
                    problemFields: ["format"],
                }
            }
        }
    }

    if (!parsed.format) {
        // If either topicOptions or formats are null, it means something went wrong with allTopicsFormatsOptions.
        // This is an "internal" issue and we should get pinged for it.
        // TODO: Can we add some sort of tracking like Sentry or CloudWatch for this?
        return {
            error: `Unable to determine formats for topic [ ${topic} ]. This is an internal issue; please contact Atlas devs if you see this.`,
            problemFields: ["topic", "format"],
        }
    }

    return {};
}

function processCoStreamSettings(item: EventImportData, parsed: Partial<AtlasEvent>): ParseResult {
    if (item.costreaming_settings) {
        parsed.costreaming_settings = fuzzySelect(item.costreaming_settings, coStreamingSettingsOptions);
    } else {
        parsed.costreaming_settings = coStreamingSettingsOptions[0];
    }

    return {};
}

function processPCCId(parsed: Partial<AtlasEvent>, currentContentCreatorID: string): ParseResult {
    parsed.premium_content_creator_id = +currentContentCreatorID;
    if (!parsed.premium_content_creator_id) {
        return {
            error: "Unable to properly set premium_content_creator_id; this may indicate a bug with CSV uploads.",
        };
    }

    return {};
}
