CREATE EXTENSION IF NOT EXISTS "uuid-ossp";

CREATE FUNCTION update_updated_at_utc_column()
    RETURNS trigger AS $$
BEGIN
  NEW.updated_at_utc = current_timestamp;
RETURN NEW;
END;
$$ LANGUAGE 'plpgsql';

CREATE TABLE IF NOT EXISTS achievements(
    id uuid PRIMARY KEY DEFAULT uuid_generate_v4(),
    key varchar(255) NOT NULL,
    progress_cap integer NOT NULL,
    img varchar(255) NOT NULL,
    img_sm varchar(255) NOT NULL,
    img_2x varchar(255) NOT NULL,
    img_3x varchar(255) NOT NULL,

    created_at_utc timestamp DEFAULT current_timestamp NOT NULL,
    updated_at_utc timestamp,

    CONSTRAINT achievements_key_not_empty CHECK (key <> ''),
    CONSTRAINT achievements_progress_cap_is_positive CHECK (progress_cap > 0)
    );

CREATE UNIQUE INDEX achievements_key_progress_cap_uniq_idx ON achievements (key, progress_cap);
CREATE INDEX achievements_key_idx ON achievements (key);

CREATE TABLE IF NOT EXISTS progressions(
    id uuid PRIMARY KEY DEFAULT uuid_generate_v4(),
    channel_id varchar(255) NOT NULL,
    achievement_id uuid REFERENCES achievements (id),
    progress integer DEFAULT 0 NOT NULL,

    created_at_utc timestamp DEFAULT current_timestamp NOT NULL,
    updated_at_utc timestamp,
    completed_at_utc timestamp,

    CONSTRAINT progressions_channel_id_not_empty CHECK (channel_id <> ''),
    CONSTRAINT progressions_progress_is_non_negative CHECK (progress >= 0)
    );

CREATE UNIQUE INDEX progressions_channel_id_achievement_id_uniq_idx ON progressions (channel_id, achievement_id);
CREATE INDEX progressions_channel_id_idx ON progressions (channel_id);

CREATE TABLE IF NOT EXISTS quests(
    id uuid PRIMARY KEY DEFAULT uuid_generate_v4(),
    key varchar(255) NOT NULL,

    created_at_utc timestamp DEFAULT current_timestamp NOT NULL,
    updated_at_utc timestamp
    );

CREATE UNIQUE INDEX quests_key_uniq_idx ON quests (key);

CREATE TABLE IF NOT EXISTS quest_achievement_rankings(
    id uuid PRIMARY KEY DEFAULT uuid_generate_v4(),
    quest_id uuid REFERENCES quests (id) NOT NULL,
    achievement_id uuid REFERENCES achievements (id) NOT NULL,
    rank integer NOT NULL,

    created_at_utc timestamp DEFAULT current_timestamp NOT NULL,
    updated_at_utc timestamp,

    CONSTRAINT quest_achievement_rankings_rank_is_positive CHECK (rank >= 0)
    );

CREATE UNIQUE INDEX quest_achievement_rankings_quest_id_achievement_id_uniq_idx ON quest_achievement_rankings (quest_id, achievement_id);
CREATE UNIQUE INDEX quest_achievement_rankings_quest_id_rank_uniq_idx ON quest_achievement_rankings (quest_id, rank);

CREATE TRIGGER achievements_before_update
    BEFORE UPDATE ON achievements
    FOR EACH ROW EXECUTE PROCEDURE update_updated_at_utc_column();

CREATE TRIGGER progressions_before_update
    BEFORE UPDATE ON progressions
    FOR EACH ROW EXECUTE PROCEDURE update_updated_at_utc_column();

CREATE TRIGGER quests_before_update
    BEFORE UPDATE ON quests
    FOR EACH ROW EXECUTE PROCEDURE update_updated_at_utc_column();

CREATE TRIGGER quest_achievement_rankings_before_update
    BEFORE UPDATE ON quest_achievement_rankings
    FOR EACH ROW EXECUTE PROCEDURE update_updated_at_utc_column();

-- quest status
CREATE TABLE IF NOT EXISTS quest_achievement_ranking_statuses(
    id uuid PRIMARY KEY DEFAULT uuid_generate_v4(),
    channel_id varchar(255) NOT NULL,
    quest_achievement_ranking_id uuid REFERENCES quest_achievement_rankings (id) NOT NULL,

    created_at_utc timestamp DEFAULT current_timestamp NOT NULL,
    updated_at_utc timestamp,

    CONSTRAINT quest_achievement_ranking_statuses_channel_id_not_empty CHECK (channel_id <> '')
    );

CREATE UNIQUE INDEX quest_achievement_ranking_statuses_channel_id_quest_achievement_ranking_id_uniq_idx ON quest_achievement_ranking_statuses (channel_id, quest_achievement_ranking_id);
CREATE TRIGGER quest_achievement_ranking_statuses_before_update
    BEFORE UPDATE ON quest_achievement_ranking_statuses
    FOR EACH ROW EXECUTE PROCEDURE update_updated_at_utc_column();

CREATE TABLE IF NOT EXISTS quests_status(
    quest_id uuid REFERENCES quests (id) NOT NULL,
    channel_id varchar(255) NOT NULL,

    created_at_utc timestamp DEFAULT current_timestamp NOT NULL,
    updated_at_utc timestamp);

CREATE UNIQUE INDEX quests_status_quest_id_channel_id_uniq_idx ON quests_status (quest_id, channel_id);

CREATE TRIGGER quests_status_before_update
    BEFORE UPDATE ON quests_status
    FOR EACH ROW EXECUTE PROCEDURE update_updated_at_utc_column();

CREATE INDEX quests_status_channel_id_idx ON quests_status (channel_id);
CREATE INDEX quests_status_quest_idx ON quests_status (quest_id);

CREATE INDEX quest_achievement_ranking_statuses_channel_id_idx ON quest_achievement_ranking_statuses (channel_id);
CREATE INDEX quest_achievement_ranking_statuses_quest_achievement_ranking_id_idx ON quest_achievement_ranking_statuses (quest_achievement_ranking_id);

ALTER TABLE achievements ADD COLUMN enabled boolean;
UPDATE achievements SET enabled = 't';
ALTER TABLE achievements ALTER COLUMN enabled SET NOT NULL;
ALTER TABLE achievements ALTER COLUMN enabled SET DEFAULT TRUE;



