import * as cdk from "@aws-cdk/core";
import * as ec2 from "@aws-cdk/aws-ec2";
import * as ecs from "@aws-cdk/aws-ecs";
import * as iam from "@aws-cdk/aws-iam";
import * as lambdaGo from "@aws-cdk/aws-lambda-go";
import path = require("path");
import { VPCStack } from "../vpc-stack";

interface CleanTapValidatorProps extends cdk.StackProps {
  vpcStack: VPCStack;
}

export class CleanTapValidatorStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: CleanTapValidatorProps) {
    super(scope, id, props);

    const cluster = new ecs.Cluster(this, "Cluster", {
      vpc: props.vpcStack.vpc,
      containerInsights: true,
    });

    const role = new iam.Role(this, "Role", {
      assumedBy: new iam.ServicePrincipal("ecs-tasks.amazonaws.com"),
      managedPolicies: [
        "AmazonRedshiftFullAccess",
        "SecretsManagerReadWrite",
      ].map((name) => iam.ManagedPolicy.fromAwsManagedPolicyName(name)),
    });
    role.addToPolicy(
      new iam.PolicyStatement({
        resources: ["*"],
        actions: ["sts:*"],
      }),
    );

    const taskDefinition = new ecs.FargateTaskDefinition(this, "Task", {
      family: id,
      taskRole: role,
      cpu: 1024,
      memoryLimitMiB: 3072
    });

    const container = taskDefinition.addContainer("Container", {
      image: ecs.ContainerImage.fromAsset(
        path.join(__dirname, "../../../validator/task"),
      ),
      memoryReservationMiB: 3072,
      logging: new ecs.AwsLogDriver({
        logRetention: 30,
        streamPrefix: id,
      }),
      environment: {
        AWS_REGION: cdk.Stack.of(this).region,
        ENVIRONMENT: "staging",
      },
    });

    container.addUlimits({
      name: ecs.UlimitName.NOFILE,
      softLimit: 65_536,
      hardLimit: 65_536,
    });

    new lambdaGo.GoFunction(this, "Fn", {
      vpc: props.vpcStack.vpc,
      allowPublicSubnet: true,
      entry: path.join(__dirname, "../../../validator/runner"),
      retryAttempts: 0,
      description:
        "Runs a validator task in the ECS Clean Tap Validate cluster",
      timeout: cdk.Duration.minutes(15),
      logRetention: 30,
      initialPolicy: [
        new iam.PolicyStatement({
          actions: ["ecs:*", "iam:GetRole", "iam:PassRole"],
          resources: ["*"],
        }),
      ],
      environment: {
        REGION: cdk.Stack.of(this).region,
        CLUSTER_ARN: cluster.clusterArn,
        CONTAINER_NAME: container.containerName,
        TASK_DEFINITION: taskDefinition.taskDefinitionArn,
        SUBNET_IDS: JSON.stringify(
          props.vpcStack.privateSubnets.subnets!.map(
            (subnet) => subnet.subnetId,
          ),
        ),
      },
    });
  }
}
