import * as cdk from "@aws-cdk/core";
import * as ec2 from "@aws-cdk/aws-ec2";
import * as redshift from "@aws-cdk/aws-redshift";
import * as iam from "@aws-cdk/aws-iam";
import { VPCStack } from "../vpc-stack";

interface RedshiftStackProps extends cdk.StackProps {
  vpcStack: VPCStack;
  clusterName: string;
}

export class RedshiftStack extends cdk.Stack {
  public readonly role: iam.IRole;
  public readonly cluster: redshift.Cluster;

  constructor(scope: cdk.Construct, id: string, props: RedshiftStackProps) {
    super(scope, id, props);

    this.role = new iam.Role(this, "Role", {
      assumedBy: new iam.ServicePrincipal("redshift.amazonaws.com"),
      description: "Role for the Achievements service's Tahoe Tap",
      managedPolicies: [
        // Just for testing, so we may unload to S3 to verify data after querying
        iam.ManagedPolicy.fromAwsManagedPolicyName("AmazonS3FullAccess"),
      ],
    });

    const parameterGroup = new redshift.ClusterParameterGroup(
      this,
      "ParameterGroup",
      {
        parameters: {
          wlm_json_configuration: '[{"query_concurrency":5}]',
          require_ssl: "true",
        },
      },
    );
    const cluster = new redshift.Cluster(this, "Cluster", {
      clusterName: props.clusterName,
      vpc: props.vpcStack.vpc,
      vpcSubnets: props.vpcStack.publicSubnets,
      masterUser: {
        masterUsername: "tahoeadmin",
      },
      defaultDatabaseName: "product",
      roles: [this.role],
      numberOfNodes: 5,
      nodeType: redshift.NodeType.RA3_16XLARGE,
      parameterGroup: parameterGroup,
    });

    cluster.connections.allowFrom(
      ec2.Peer.ipv4(props.vpcStack.vpc.vpcCidrBlock),
      ec2.Port.tcp(5439),
    );

    this.cluster = cluster;
  }
}
