import cdk = require("@aws-cdk/core");
import ec2 = require("@aws-cdk/aws-ec2");

const privateSubnetIDs = [
  "subnet-6f4a7808",
  "subnet-45e7ad0c",
  "subnet-86fe23dd",
];

const publicSubnetIDs = [
  "subnet-62497b05",
  "subnet-6ffc2134",
  "subnet-58fab011",
]

interface VPCStackProps extends cdk.StackProps {
  vpcID: string;
}

export class VPCStack extends cdk.Stack {
  public vpc: ec2.IVpc;
  /**
   * Private subnets in the twitch-cb-aws account has public IP addresses
   * so are not marked and detected as Private automatically. We specify
   * them manually by subnetID
   */
  public privateSubnets: ec2.SubnetSelection;
  public publicSubnets: ec2.SubnetSelection;

  constructor(scope: cdk.Construct, id: string, props: VPCStackProps) {
    super(scope, id, {
      description: "The VPC for shared resources in the twitch-cb-aws account",
      ...props,
    });

    this.vpc = ec2.Vpc.fromLookup(this, "VPC", {
      vpcId: props.vpcID,
    });

    this.privateSubnets = {
      subnets: privateSubnetIDs.map((subnetID, index) =>
        ec2.Subnet.fromSubnetAttributes(this, `PrivateSubnet${index}`, {
          subnetId: subnetID,
        }),
      ),
    };

    this.publicSubnets = {
      subnets: publicSubnetIDs.map((subnetID, index) =>
        ec2.Subnet.fromSubnetAttributes(this, `PublicSubnet${index}`, {
          subnetId: subnetID,
        }),
      ),
    };
  }
}
