package achievements

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type getAchievementsSuite struct {
	suite.Suite
}

func (s *getAchievementsSuite) TestFailure() {
	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetAchievements(context.Background(), "123", "456", nil)
	s.NotNil(err)
	s.Nil(resp)
}

func (s *getAchievementsSuite) TestMalformedResponse() {
	payload := `
		{
			"status": 200,
			"message": "malformed JSON String with extra double-quotes""
		}
	`

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)
	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetAchievements(context.Background(), "123", "456", nil)
	s.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	s.NoError(err)
	s.NotEqual(string(jsonResponse), payload)
}

func (s *getAchievementsSuite) TestSuccess() {
	channelID := "123"
	viewerID := "456"

	payload := `
		{
			"achievements": [{
				"id": "id",
				"key": "key",
				"progress": 8999,
				"progress_cap": 9000,
				"completed_at": null
			}]
		}
	`

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		s.Equal(fmt.Sprintf("/v1/internal/channel/%s/viewer/%s/achievements", channelID, viewerID), r.URL.EscapedPath())

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)
	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetAchievements(context.Background(), channelID, viewerID, nil)
	s.NoError(err)

	if s.NotNil(resp) {
		if s.Len(resp.Achievements, 1) {
			achievement := resp.Achievements[0]

			s.Equal("id", achievement.ID)
			s.Equal("key", achievement.Key)
			s.Equal(8999, achievement.Progress)
			s.Equal(9000, achievement.ProgressCap)
			s.Nil(achievement.CompletedAt)
		}
	}
}

func TestGetAchievementsSuite(t *testing.T) {
	suite.Run(t, &getAchievementsSuite{})
}
