package achievements

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/cb/achievements/view"

	"code.justin.tv/foundation/twitchclient"

	"github.com/stretchr/testify/suite"
)

type postV1StreamManagerVisitedSuite struct {
	suite.Suite
}

func (s *postV1StreamManagerVisitedSuite) TestFailure() {
	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.PostV1StreamManagerVisited(context.Background(), "123", "123", nil)
	s.NotNil(err)
	s.Nil(resp)
}

func (s *postV1StreamManagerVisitedSuite) TestMalformedResponse() {
	payload := `
		{
			"status": 200,
			"message": "malformed JSON String with extra double-quotes""
		}
	`

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.PostV1StreamManagerVisited(context.Background(), "123", "123", nil)
	s.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	s.NoError(err)
	s.NotEqual(string(jsonResponse), payload)
}

func (s *postV1StreamManagerVisitedSuite) TestSuccess() {
	channelID := "123"
	viewerID := "123"

	payload := `
		{
			"status": 200,
			"message": "",
			"meta": {},
			"data": {}
		}
	`

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		s.Equal(fmt.Sprintf("/v1/internal/channel/%s/viewer/%s/streamManagerVisited", channelID, viewerID), r.URL.EscapedPath())

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	expectedResp := &view.V1StreamManagerVisitedResponse{
		Status:  200,
		Message: "",
		Meta:    &view.V1StreamManagerVisitedMeta{},
		Data:    &view.V1StreamManagerVisitedData{},
	}

	resp, err := client.PostV1StreamManagerVisited(context.Background(), channelID, viewerID, nil)
	s.NoError(err)
	s.Equal(expectedResp, resp)
}

func TestPostV1StreamManagerVisitedSuite(t *testing.T) {
	suite.Run(t, &postV1StreamManagerVisitedSuite{})
}
