package main

import (
	"net/http"
	"os"
	"strconv"
	"time"

	"code.justin.tv/cb/achievements/internal/clients/ripley"

	"code.justin.tv/cb/achievements/internal/clients/users"

	"code.justin.tv/cb/achievements/internal/clients/sns"

	"code.justin.tv/cb/achievements/config"
	"code.justin.tv/cb/achievements/internal/api"
	"code.justin.tv/cb/achievements/internal/auth"
	"code.justin.tv/cb/achievements/internal/clients/dart"
	"code.justin.tv/cb/achievements/internal/clients/db"
	"code.justin.tv/cb/achievements/internal/clients/dynamo"
	"code.justin.tv/cb/achievements/internal/clients/moneypenny"
	"code.justin.tv/cb/achievements/internal/clients/stats"
	"code.justin.tv/cb/achievements/internal/httputil"
	log "github.com/sirupsen/logrus"
)

const (
	readTimeout  = 5 * time.Second
	writeTimeout = 5 * time.Second
)

func init() {
	log.SetFormatter(&log.TextFormatter{
		FullTimestamp: true,
	})

	config.SetupRollbarLogging()
	config.Load()
}

func main() {
	env := config.Environment

	decoder, err := auth.NewDecoder(config.Values.ECCKeyPath)
	if err != nil {
		log.WithError(err).Fatal("api: failed to instantiate authorization decoder")
		return
	}

	var maxIdleConns int
	if conns := os.Getenv("DB_CONNS"); conns != "" {
		maxIdleConns, _ = strconv.Atoi(conns)
	}

	dbReader, err := db.OpenConnection(db.Config{
		Credentials:  config.Secrets.DB.Credentials,
		Address:      config.Values.DB.Addresses.Replica,
		MaxIdleConns: maxIdleConns,
	})
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"address": config.Values.DB.Addresses.Replica,
		}).Fatal("api: failed to open connection to replica database")
		return
	}

	statsClient, err := stats.NewClient(config.Values.Statsd.Host, env, "api")
	if err != nil {
		log.WithError(err).Fatal("api: failed to instantiate statsd")
		return
	}

	dynamoClient, err := dynamo.NewClient(env, config.Values.AWSRegion)
	if err != nil {
		log.WithError(err).Fatal("api: failed to instantiate new dynamodb client")
		return
	}

	moneypennyClient, err := moneypenny.NewHTTPClient(config.Values.ExternalServices.Moneypenny)
	if err != nil {
		log.WithError(err).Fatal("api: failed to instantiate moneypenny client")
		return
	}

	dartClient := dart.NewClient(config.Values.ExternalServices.Dart)
	ripleyClient := ripley.NewClient(config.Values.ExternalServices.Ripley)

	snsClient, err := sns.NewClient(env, config.Values.AWSRegion, config.Values.AchievementSNSTopics.StreamManagerVisited)
	if err != nil {
		log.WithError(err).Fatal("api: failed to instantiate new sns client")
		return
	}

	usersServiceClient, err := users.NewUserServiceClient(config.Values.ExternalServices.Users)
	if err != nil {
		log.WithError(err).Fatal("api: failed to instantiate new users client")
		return
	}

	log.Info("api: application starting in ", env)

	server := &http.Server{
		Addr: ":8000",
		Handler: api.NewServer(&api.ServerParams{
			Auth:       decoder,
			DBReader:   dbReader,
			Statsd:     statsClient,
			DynamoDB:   dynamoClient,
			Sns:        snsClient,
			Moneypenny: moneypennyClient,
			Notifier:   dartClient,
			Users:      usersServiceClient,
			Ripley:     ripleyClient,
		}),
		ReadTimeout:  readTimeout,
		WriteTimeout: writeTimeout,
	}

	go func() {
		log.Info("api: server listening on http://localhost", server.Addr)

		if err := server.ListenAndServe(); err != http.ErrServerClosed {
			log.WithError(err).Fatal("api: server failed fatally while listening")
		}
	}()

	httputil.Graceful(server)
}
