package main

import (
	"net/http"

	"code.justin.tv/cb/achievements/config"
	"code.justin.tv/cb/achievements/internal/clients/db"
	"code.justin.tv/cb/achievements/internal/clients/dynamo"
	"code.justin.tv/cb/achievements/internal/clients/eventbus"
	"code.justin.tv/cb/achievements/internal/clients/redshift"
	"code.justin.tv/cb/achievements/internal/clients/sandstorm"
	"code.justin.tv/cb/achievements/internal/clients/sqs"
	"code.justin.tv/cb/achievements/internal/clients/stats"
	"code.justin.tv/cb/achievements/internal/clients/twitchcon"
	"code.justin.tv/cb/achievements/internal/httputil"
	"code.justin.tv/cb/achievements/internal/sourcer"
	eventbusClient "code.justin.tv/eventbus/client"
	userfollow "code.justin.tv/eventbus/schema/pkg/user_follow_user"
	_ "github.com/lib/pq"
	log "github.com/sirupsen/logrus"
)

func init() {
	log.SetFormatter(&log.TextFormatter{
		FullTimestamp: true,
	})

	config.SetupRollbarLogging()
	config.Load()
	config.LoadTahoeReplicaSecret(&config.Values)
}

func main() {
	env := config.Environment
	region := config.Values.AWSRegion

	sqsClient, err := sqs.NewClient(env, region, config.Values.WorkerSQSQueueURL)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to instantiate new sqs client")
		return
	}

	dynamoClient, err := dynamo.NewClient(env, region)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to instantiate new dynamodb client")
		return
	}

	statsClient, err := stats.NewClient(config.Values.Statsd.Host, config.Environment, "sourcer")
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to instantiate statsd")
		return
	}

	sandstormManager, err := sandstorm.NewManager(config.Values.Sandstorm.RoleARN, region)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to create sandstorm manager")
		return
	}

	dirtyTahoePassword, err := sandstormManager.GetSecret(config.Values.Sandstorm.SecretNames.DirtyTahoeReplicaPassword)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to retrieve tahoe password from sandstorm")
		return
	}

	dirtyTahoeReplica, err := redshift.NewClient(&redshift.Config{
		Host:     config.Values.DirtyTahoeReplica.Host,
		User:     config.Values.DirtyTahoeReplica.User,
		Password: dirtyTahoePassword,
		Port:     config.Values.DirtyTahoeReplica.Port,
		DBName:   config.Values.DirtyTahoeReplica.DBName,
	}, statsClient)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to instantiate new redshift client for dirty tahoe replica")
		return
	}

	redshiftPassword, err := sandstormManager.GetSecret(config.Values.Sandstorm.SecretNames.RedshiftPassword)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to retrieve tahoe password from sandstorm")
		return
	}

	redshiftDB, err := redshift.NewClient(&redshift.Config{
		Host:     config.Values.Redshift.Host,
		User:     config.Values.Redshift.User,
		Password: redshiftPassword,
		Port:     config.Values.Redshift.Port,
		DBName:   config.Values.Redshift.DBName,
	}, statsClient)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to instantiate new redshift client")
		return
	}

	tahoeReplica, err := redshift.NewClient(&redshift.Config{
		Host:     config.Values.TahoeReplica.Host,
		User:     config.Values.TahoeReplica.User,
		Password: config.Values.TahoeReplica.Password,
		Port:     config.Values.TahoeReplica.Port,
		DBName:   config.Values.TahoeReplica.DBName,
	}, statsClient)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to instantiate new redshift client for clean tahoe replica")
		return
	}

	sqsStatsClient, err := stats.NewClient(config.Values.Statsd.Host, config.Environment, "sqs")
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to instantiate statsd")
		return
	}

	twitchconPurchasersToken, err := sandstormManager.GetSecret(config.Values.Sandstorm.SecretNames.TwitchconBearerToken)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to retrieve twitchcon bearer token from sandstorm")
		return
	}

	twitchconClient := twitchcon.NewClient(config.Values.Twitchcon.URL, twitchconPurchasersToken)

	dbClient, err := db.OpenConnection(db.Config{
		Credentials: config.Secrets.DB.Credentials,
		Address:     config.Values.DB.Addresses.Master,
	})
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to open connection to database")
		return
	}

	userFollowUserHandler := eventbus.NewUserFollowUserHandler(sqsClient, sqsStatsClient)

	mux := eventbusClient.NewMux()
	userfollow.RegisterCreateHandler(mux, userFollowUserHandler.ProcessNFollowersUpdate)

	ebClient, err := eventbus.NewEventbusClient(config.Environment, region, config.Values.EventbusSQSQueueURL, mux)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to setup eventbus client")
		return
	}
	defer func() {
		err = ebClient.Shutdown()
		if err != nil {
			log.WithError(err).Fatal("sourcer: unable to gracefully shutdown eventbus client")
		}
	}()

	server := &http.Server{
		Addr: ":8000",
		Handler: sourcer.NewServer(&sourcer.ServerParams{
			SQS:               sqsClient,
			DynamoDB:          dynamoClient,
			TahoeReplica:      tahoeReplica,
			Redshift:          redshiftDB,
			DirtyTahoeReplica: dirtyTahoeReplica,
			Statsd:            statsClient,
			SQSStatsd:         sqsStatsClient,
			Twitchcon:         twitchconClient,
			DBWriter:          dbClient,
		}),
	}

	go func() {
		log.Info("sourcer: server listening on http://localhost", server.Addr)

		if err := server.ListenAndServe(); err != http.ErrServerClosed {
			log.WithError(err).Fatal("sourcer: server failed fatally while listening")
		}
	}()

	httputil.Graceful(server)
}
