package main

import (
	"net/http"
	"os"

	"code.justin.tv/cb/achievements/config"
	"code.justin.tv/cb/achievements/internal/clients/dart"
	"code.justin.tv/cb/achievements/internal/clients/db"
	"code.justin.tv/cb/achievements/internal/clients/dynamo"
	"code.justin.tv/cb/achievements/internal/clients/sns"
	"code.justin.tv/cb/achievements/internal/clients/stats"
	"code.justin.tv/cb/achievements/internal/clients/tracking"
	"code.justin.tv/cb/achievements/internal/httputil"
	"code.justin.tv/cb/achievements/internal/worker"
	_ "github.com/lib/pq"
	log "github.com/sirupsen/logrus"
)

func init() {
	log.SetFormatter(&log.TextFormatter{
		FullTimestamp: true,
	})

	config.SetupRollbarLogging()
	config.Load()
}

func main() {
	env := config.Environment
	region := config.Values.AWSRegion

	snsClient, err := sns.NewClient(env, region, config.Values.NotificationSNSTopics.Achievement)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to instantiate new sns client")
		return
	}

	dynamoClient, err := dynamo.NewClient(env, region)
	if err != nil {
		log.WithError(err).Fatal("sourcer: failed to instantiate new dynamodb client")
		return
	}

	dbDisabled := os.Getenv("DB_DISABLED") == "true"

	var dbClient *db.Client
	if !dbDisabled {
		dbClient, err = db.OpenConnection(db.Config{
			Credentials: config.Secrets.DB.Credentials,
			Address:     config.Values.DB.Addresses.Master,
		})
		if err != nil {
			log.WithError(err).Fatal("worker: failed to open connection to database")
			return
		}
	}

	statsClient, err := stats.NewClient(config.Values.Statsd.Host, config.Environment, "worker")
	if err != nil {
		log.WithError(err).Fatal("worker: failed to instantiate statsd")
		return
	}

	sqsStatsClient, err := stats.NewClient(config.Values.Statsd.Host, config.Environment, "sqs")
	if err != nil {
		log.WithError(err).Fatal("worker: failed to instantiate statsd")
		return
	}

	dartClient := dart.NewClient(config.Values.ExternalServices.Dart)

	trackingClient, err := tracking.NewClient()
	if err != nil {
		log.WithError(err).Fatal("worker: failed to instantiate tracking client")
		return
	}

	defer func() {
		trackingClient.Close()
	}()

	server := &http.Server{
		Addr: ":8000",
		Handler: worker.NewServer(&worker.ServerParams{
			DBDisabled: dbDisabled,
			DBReader:   dbClient,
			DBWriter:   dbClient,
			Statsd:     statsClient,
			SQSStatsd:  sqsStatsClient,
			SNS:        snsClient,
			DynamoDB:   dynamoClient,
			Notifier:   dartClient,
			Tracking:   trackingClient,
		}),
	}

	go func() {
		log.Info("worker: server listening on http://localhost", server.Addr)

		if err := server.ListenAndServe(); err != http.ErrServerClosed {
			log.WithError(err).Fatal("worker: server failed fatally while listening")
		}
	}()

	httputil.Graceful(server)
}
