package config

import (
	"context"
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"runtime"
	"strconv"
	"strings"

	"code.justin.tv/cb/achievements/env"
	secretsmanager "code.justin.tv/cb/achievements/internal/clients/secrets"
	"github.com/heroku/rollrus"
	log "github.com/sirupsen/logrus"
	"gopkg.in/yaml.v2"
)

const (
	configPath = "/var/app/current/config.yaml"

	// The application secrets YAML path is defined in the Elastic Beanstalk
	// extensions for the Sandstorm-Agent:
	//
	// .ebextensions/data/sandstorm-agent/conf.d/03-secrets.conf
	// .ebextensions/data/sandstorm-agent/templates.d/secrets.yaml
	secretsPath = "/var/app/secrets.yaml"

	// The Rollbar token path is defined in the Elastic Beanstalk
	// extensions for the Sandstorm-Agent:
	//
	// .ebextensions/data/sandstorm-agent/conf.d/02-rollbar.conf
	// .ebextensions/data/sandstorm-agent/templates.d/rollbar-token
	rollbarTokenPath = "/var/app/rollbar_token"
)

var (
	// Environment is a global variable of the application environment
	// according to the environment variable `ENVIRONMENT`.
	Environment string

	// Values is a global variable containing the fields of all
	// environment-specific configuration values for the application.
	Values values

	// Secrets is a global variable containing the fields of all
	// environment-specific secrets for the application.
	Secrets secrets
)

// values contains the non-secret configuration fields for the application.
//
// After defining a field within the values structure,
// the attribute MUST be defined in the following YAML files
// for decoding during application start-up:
//
// * config/development.yaml
// * config/staging.yaml
// * config/production.yaml
//
// Example:
//
// type values struct {
// 	DB struct {
// 		Addresses struct {
// 			Master  string
// 			Replica string
// 		}
// 	}
// }
//
// development.yaml:
//   db:
//     addresses:
//       master: http://0.0.0.0
//       replica: http://1.2.3.4
type values struct {
	AchievementSNSTopics struct {
		FirstStream          string `yaml:"first_stream"`
		StreamManagerVisited string `yaml:"stream_manager_visited"`
		NAutohost            string `yaml:"n_autohost"`
		ChannelUpdate        string `yaml:"channel_update"`
		OnboardingComplete   string `yaml:"onboarding_complete"`
		OnboardingCancel     string `yaml:"onboarding_cancel"`
	} `yaml:"achievements"`

	NotificationSNSTopics struct {
		Achievement string `yaml:"achievement_sns_arn"`
		Quest       string `yaml:"quest_sns_arn"`
	} `yaml:"completion_notification"`

	ExternalServices struct {
		Dart       string `yaml:"dart"`
		Moneypenny string `yaml:"moneypenny"`
		Users      string `yaml:"usersservice"`
		Ripley     string `yaml:"ripley"`
	} `yaml:"external_services"`

	AWSRegion string `yaml:"aws_region"`

	DB struct {
		Addresses struct {
			Master  string
			Replica string
		}
	}

	DynamoDB struct {
		Tables struct {
			FirstDashboardAccesses string `yaml:"first_dashboard_accesses"`
			FirstStreams           string `yaml:"first_streams"`
			ChannelUpdates         string `yaml:"channel_updates"`
		} `yaml:"tables"`
	} `yaml:"dynamodb"`

	ECCKeyPath string `yaml:"ecc_key_path"`

	Redshift struct {
		Host   string `yaml:"host"`
		User   string `yaml:"user"`
		Port   string `yaml:"port"`
		DBName string `yaml:"dbname"`
	} `yaml:"redshift"`

	DirtyTahoeReplica struct {
		Host   string `yaml:"host"`
		User   string `yaml:"user"`
		Port   string `yaml:"port"`
		DBName string `yaml:"dbname"`
	} `yaml:"dirty_tahoe_replica"`

	Statsd struct {
		Host string `yaml:"host"`
	} `yaml:"statsd"`

	Sandstorm struct {
		RoleARN     string `yaml:"role_arn"`
		SecretNames struct {
			DirtyTahoeReplicaPassword string `yaml:"dirty_tahoe_replica_password"`
			RedshiftPassword          string `yaml:"redshift_password"`
			DBCredentials             string `yaml:"db_credentials"`
			TwitchconBearerToken      string `yaml:"twitchcon_bearer_token"`
		} `yaml:"secret_names"`
	} `yaml:"sandstorm"`

	WorkerSQSQueueURL   string `yaml:"worker_sqs_queue_url"`
	EventbusSQSQueueURL string `yaml:"eventbus_sqs_queue_url"`

	Twitchcon struct {
		URL string `yaml:"url"`
	} `yaml:"twitchcon"`

	TahoeReplica struct {
		SecretID string `yaml:"secret_id"`
		Host     string `yaml:"-"`
		User     string `yaml:"-"`
		Password string `yaml:"-"`
		Port     string `yaml:"-"`
		DBName   string `yaml:"-"`
	} `yaml:"tahoe_replica"`
}

// secrets contains the secret configuration values for the application.
//
// For development ONLY, after defining a field in the secrets structure,
// the attribute MUST be defined in config/development.yaml.
//
// For staging or production, a secret's attribute key and template value
// MUST be added to the following file:
//
// .ebextensions/data/sandstorm-agent/templates.d/secrets
//
// for the Sandstorm-Agent to insert first and then for unmarshaling
// during the start-up of the application.
// The secret MUST also be created within Sandstorm under the same key.
//
// Example:
//
// type secrets struct {
// 	DB struct{
// 		Password string
// 	}
// }
//
// development.yaml:
//   db:
//     password: hunter2
//
// .ebextensions/data/sandstorm-agent/templates.d/secrets:
//   db:
//     password: {{ key "creator-business/achievements/{ENVIRONMENT}/password" }}
type secrets struct {
	DB struct {
		Credentials string
	}
}

func init() {
	Environment = os.Getenv("ENVIRONMENT")
	if Environment == "" {
		Environment = env.Development
	}
}

// SetupRollbarLogging reads the Rollbar access token from file,
// and adds a hook to Logrus for reporting the logs of the preset
// levels to Rollbar.
func SetupRollbarLogging() {
	buf, err := ioutil.ReadFile(rollbarTokenPath)
	if err != nil {
		log.Warn("No Rollbar access token found for ", Environment)
		log.Warn("Logs will not be sent to Rollbar")
		return
	}

	rollbarLevels := []log.Level{
		log.ErrorLevel,
		log.FatalLevel,
		log.PanicLevel,
		log.WarnLevel,
		log.DebugLevel,
	}

	log.SetLevel(log.DebugLevel)
	rollrus.SetupLoggingForLevels(strings.TrimSpace(string(buf)), Environment, rollbarLevels)
	log.Info("Logs set to send to Rollbar")
}

// Load decodes both non-secret and secret configuration values from YAML.
//
// In development, the `config/development.yaml` file is loaded to
// populate BOTH the Values and Secrets global variables.
//
// In staging or production, the corresponding non-secret config YAML
// (config/staging.yaml or config/production.yaml) is added
// to the /var/app/current directory as config.yaml during
// the deployment process (See: scripts/deploy_beanstalk.sh).
//
// The secret config YAML is created by the Sandstorm-Agent
// at /var/app/secrets.yaml
// (See: .ebextensions/data/sandstorm-agent/conf.d/03-secrets.conf)
func Load() {
	configFilePath := configPath
	secretsFilePath := secretsPath

	if Environment == env.Development || Environment == env.Test {
		_, filename, _, ok := runtime.Caller(0)
		if !ok {
			log.Panic("No caller information to get config file name")
		}
		localConfigPath := map[string]string{env.Development: "development.yaml", env.Test: "test.yaml"}

		filePath := fmt.Sprint(path.Dir(filename), "/", localConfigPath[Environment])
		configFilePath = filePath
		secretsFilePath = filePath
	}

	Values = loadValues(configFilePath)
	Secrets = loadSecrets(secretsFilePath)
}

// LoadTahoeReplicaSecret loads TahoeTap secret from secretsmanager
func LoadTahoeReplicaSecret(v *values) {
	if v == nil {
		log.Panic("global value, v is nil")
	}

	if v.TahoeReplica.SecretID == "" {
		log.Panic("secretID is required")
	}

	secretsClient, err := secretsmanager.NewClient(Environment, v.AWSRegion)
	if err != nil {
		log.WithError(err).Panic("Failed to instantiate new secrets client")
	}

	tahoeReplicaSecret, err := secretsClient.GetTahoeReplicaSecret(context.Background(), v.TahoeReplica.SecretID)
	if err != nil {
		log.WithError(err).Panic("Failed to get tahoe tap secret from secretsmanager")
	}

	v.TahoeReplica.User = tahoeReplicaSecret.Username
	v.TahoeReplica.Password = tahoeReplicaSecret.Password
	v.TahoeReplica.Host = tahoeReplicaSecret.Host
	v.TahoeReplica.Port = strconv.Itoa(tahoeReplicaSecret.Port)
	v.TahoeReplica.DBName = tahoeReplicaSecret.DBName
}

func loadValues(filePath string) values {
	v := values{}

	log.Info("Loading config from ", filePath)

	bytes, err := ioutil.ReadFile(filePath)
	if err != nil {
		log.WithError(err).Panic("Failed to open config YAML")
	}

	err = yaml.Unmarshal(bytes, &v)
	if err != nil {
		log.WithError(err).Panic("Failed to decode config YAML")
	}

	return v
}

func loadSecrets(filePath string) secrets {
	s := secrets{}

	log.Info("Loading secrets from ", filePath)

	bytes, err := ioutil.ReadFile(filePath)
	if err != nil {
		log.WithError(err).Panic("Failed to open configuration secrets YAML")
	}

	err = yaml.Unmarshal(bytes, &s)
	if err != nil {
		log.WithError(err).Panic("Failed to decode configuration secrets YAML")
	}

	return s
}
