package achievement

import (
	"context"
	"fmt"

	"code.justin.tv/cb/achievements/internal/clients/db"
	"github.com/pkg/errors"
)

// WorkerJob instantiated in job_builder
type WorkerJob struct {
	Clients        *Clients
	AchievementJob AchJob
}

// Job represents the interface for the worker process to run
type Job interface {
	Process(ctx context.Context) (*CompletedJob, error)
}

// AchJob represents an individual achievement's interface
type AchJob interface {
	Key() string
	GetProgressions(db.AchievementGroup) []db.Progression
	ProgressRelapsable() bool
}

// Process processes a WorkerJob's achievement
func (w *WorkerJob) Process(ctx context.Context) (*CompletedJob, error) {
	progressions, err := w.AllQualifiedProgressions(ctx)
	if err != nil {
		return nil, errors.Wrap(err, fmt.Sprintf("%s error: failed to select qualified progressions by key", w.Key()))
	}

	updatedInDB, err := w.Clients.DBWriter.UpsertProgressions(ctx, progressions, w.AchievementJob.ProgressRelapsable())
	if err != nil {
		return nil, errors.Wrap(err, fmt.Sprintf("%s error: failed to upsert progressions by key", w.Key()))
	}

	return &CompletedJob{
		CompletedProgressions: db.Progressions(progressions).Completed(),
		UpdatedProgressions:   updatedInDB,
	}, nil
}

// AllQualifiedProgressions gets all qualified progressions
func (w *WorkerJob) AllQualifiedProgressions(ctx context.Context) ([]db.Progression, error) {
	achievements, err := w.Clients.DBReader.SelectAchievementsByKey(ctx, w.Key())
	if err != nil {
		return nil, err
	}

	return w.AchievementJob.GetProgressions(db.AchievementGroup(achievements)), nil
}

// Key returns the key
func (w *WorkerJob) Key() string {
	return w.AchievementJob.Key()
}

// CompletedJob is the return object of a completed job
type CompletedJob struct {
	UpdatedProgressions   []db.Progression
	CompletedProgressions []db.Progression
}

// Clients contains all client interfaces to external resources
// required by each Achievement.
type Clients struct {
	DBReader db.Reader
	DBWriter db.Writer
}
