package nautohost

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/cb/achievements/internal/achievement"
	"code.justin.tv/cb/achievements/internal/clients/db"
	log "github.com/sirupsen/logrus"
)

// Key for this job
const Key = "n_autohost"

// Input contains the required fields to process this Achievement.
type Input struct {
	ChannelID string `json:"channel_id"`
	Count     int    `json:"num_autohosted_channels"`
}

// Job is the job struct for this achievement
type Job struct {
	Clients   *achievement.Clients
	ChannelID string
	Progress  int
}

// InitJob creates new job to will handle a first stream event
func InitJob(clients *achievement.Clients, input []byte) *Job {
	parsed := Input{}
	if err := json.Unmarshal(input, &parsed); err != nil {
		msg := fmt.Sprintf("%s error: failed to initialize job", Key)

		log.WithError(err).Error(msg)
	}

	return &Job{
		Clients:   clients,
		ChannelID: parsed.ChannelID,
		Progress:  parsed.Count,
	}
}

// Key returns the key for this achievement
func (j *Job) Key() string {
	return Key
}

// GetProgressions calculates the db progressions model from the achievement group
func (j *Job) GetProgressions(group db.AchievementGroup) []db.Progression {
	// this achievement is binary -- there is no progress to be made. State-wise these
	// achievements are not started or 1 (completed)
	return group.NewChannelProgressions(j.ChannelID, j.Progress)
}

// ProgressRelapsable specifies whether the Achievement is windowed, thus the progress can go down over time.
func (j *Job) ProgressRelapsable() bool {
	return false
}
