package nconcurrents30days

import (
	"encoding/json"
	"math"

	"code.justin.tv/cb/achievements/internal/achievement"
	"code.justin.tv/cb/achievements/internal/clients/db"
	log "github.com/sirupsen/logrus"
)

// Key for this job
const Key = "n_concurrents_30_days"

// Input contains the required fields to process this Achievement.
type Input struct {
	Channels []*ChannelAverageConcurrentViewers `json:"channels"`
}

// ChannelAverageConcurrentViewers is the concurrent viewers the specified channel had
// over all sessions in the last 30 days
type ChannelAverageConcurrentViewers struct {
	ChannelID  string  `json:"channel_id"`
	AverageCCU float64 `json:"average_ccu"`
}

// Job is the job struct for this achievement
type Job struct {
	Clients  *achievement.Clients
	Channels []*ChannelAverageConcurrentViewers
}

// InitJob creates new job to will handle a first stream event
func InitJob(clients *achievement.Clients, input []byte) *Job {
	parsed := Input{}
	if err := json.Unmarshal(input, &parsed); err != nil {
		log.Error(err)
	}

	return &Job{
		Clients:  clients,
		Channels: parsed.Channels,
	}
}

// Key returns the key for this achievement
func (j *Job) Key() string {
	return Key
}

// GetProgressions calculates the db progressions model from the achievement group
func (j *Job) GetProgressions(group db.AchievementGroup) []db.Progression {
	progressions := []db.Progression{}

	for _, channel := range j.Channels {
		rounded := int(math.Floor(channel.AverageCCU))
		channelProgressions := group.NewChannelProgressions(channel.ChannelID, rounded)
		progressions = append(progressions, channelProgressions...)
	}
	return progressions
}

// ProgressRelapsable specifies whether the Achievement is windowed, thus the progress can go down over time.
func (j *Job) ProgressRelapsable() bool {
	return true
}
