package ndaysnoviewers

import (
	"encoding/json"

	"code.justin.tv/cb/achievements/internal/achievement"
	"code.justin.tv/cb/achievements/internal/clients/db"
	log "github.com/sirupsen/logrus"
)

// Key for this job
const Key = "n_days_no_viewers"

// Input contains the required fields to process this Achievement.
type Input struct {
	Channels []*ChannelConsecutiveSessionsNoViewers `json:"channels"`
}

// ChannelConsecutiveSessionsNoViewers is the number of minutes broadcast by the specified channel
// in the last 30 days
type ChannelConsecutiveSessionsNoViewers struct {
	ChannelID           string `json:"channel_id"`
	ConsecutiveSessions int    `json:"consecutive_sessions"`
}

// Job is the job struct for this achievement
type Job struct {
	Clients  *achievement.Clients
	Channels []*ChannelConsecutiveSessionsNoViewers
}

// InitJob creates new job to will handle an event
func InitJob(clients *achievement.Clients, input []byte) *Job {
	parsed := Input{}
	if err := json.Unmarshal(input, &parsed); err != nil {
		log.Error(err)
	}

	return &Job{
		Clients:  clients,
		Channels: parsed.Channels,
	}
}

// Key returns the key for this achievement
func (j *Job) Key() string {
	return Key
}

// GetProgressions calculates the db progressions model from the achievement group
func (j *Job) GetProgressions(group db.AchievementGroup) []db.Progression {
	progressions := []db.Progression{}

	for _, channel := range j.Channels {
		channelProgressions := group.NewChannelProgressions(channel.ChannelID, channel.ConsecutiveSessions)
		progressions = append(progressions, channelProgressions...)
	}
	return progressions
}

// ProgressRelapsable specifies whether the Achievement is windowed, thus the progress can go down over time.
func (j *Job) ProgressRelapsable() bool {
	return false
}
