package singlebroadcasttitle

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/cb/achievements/internal/achievement"
	"code.justin.tv/cb/achievements/internal/clients/db"
	log "github.com/sirupsen/logrus"
)

// Key is the key name of this Achievement.
// It is used to identify the type of Achievement
// in all systems.
const Key = "single_broadcast_title"

// Input contains the required fields to process this Achievement.
type Input struct {
	ChannelID string `json:"channel_id"`
}

// Job is the job struct for this achievement
type Job struct {
	Clients   *achievement.Clients
	ChannelID string
}

// InitJob creates new job to handle a first-time title-change event.
func InitJob(clients *achievement.Clients, input []byte) *Job {
	param := Input{}
	err := json.Unmarshal(input, &param)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":  Key,
			"meta": []byte(input),
		}).Error(fmt.Sprintf("%s: failed to unmarshal json", Key))
	}

	return &Job{
		Clients:   clients,
		ChannelID: param.ChannelID,
	}
}

// Key returns the key for this achievement
func (j *Job) Key() string {
	return Key
}

// GetProgressions calculates the db progressions model from the achievement group
func (j *Job) GetProgressions(group db.AchievementGroup) []db.Progression {
	// this achievement is binary -- there is no progress to be made. State-wise these
	// achievements are not started or 1 (completed)
	return group.NewChannelProgressions(j.ChannelID, 1)
}

// ProgressRelapsable specifies whether the Achievement is windowed, thus the progress can go down over time.
func (j *Job) ProgressRelapsable() bool {
	return false
}
