package api

import (
	"fmt"
	"net/http"

	"code.justin.tv/cb/achievements/internal/clients/db"
	"code.justin.tv/cb/achievements/internal/httputil"
	"code.justin.tv/cb/achievements/view"
)

// Achievements responds to a GET request and returns
// a given channel's achievement progressions (via channelID and viewerID).
func (s *Server) Achievements(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(ContextKeyChannelID).(string)
	viewerID := req.Context().Value(ContextKeyViewerID).(string)

	if channelID == "" || viewerID == "" {
		msg := fmt.Sprintf("api: viewerID or channelID is empty")
		writer.Forbidden(msg)
		return
	}

	if channelID != viewerID {
		isAdmin, err := s.users.IsAdmin(req.Context(), viewerID)
		if err != nil {
			msg := fmt.Sprintf("api: failed to determine if viewer %s is an admin", viewerID)
			writer.InternalServerError(msg, err)
			return
		}

		if !isAdmin {
			msg := fmt.Sprintf("api: unauthorized to access stream-manager-visited for channel (ID %s) as viewer ID: %s", channelID, viewerID)
			writer.Forbidden(msg)
			return
		}
	}

	dbAchievements, err := s.dbReader.SelectAllAchievementProgressionsForChannel(req.Context(), channelID)
	if err != nil {
		msg := fmt.Sprintf("Failed to query all achievement progressions for channel (ID %s)", channelID)
		writer.InternalServerError(msg, err)
		return
	}

	writer.OK(&view.Achievements{
		Achievements: dbAchievementsToView(dbAchievements),
	})
}

func dbAchievementsToView(dbAchievements []*db.Achievement) []*view.Achievement {
	var achievements []*view.Achievement

	for _, dbAchievement := range dbAchievements {
		achievement := &view.Achievement{
			ID:          dbAchievement.ID,
			Key:         dbAchievement.Key,
			ProgressCap: dbAchievement.ProgressCap,
			Level:       dbAchievement.Level,
			Image:       dbAchievement.Image,
			ImageSm:     dbAchievement.ImageSm,
			Image2x:     dbAchievement.Image2x,
			Image3x:     dbAchievement.Image3x,
		}

		if dbAchievement.Progression != nil {
			achievement.Progress = dbAchievement.Progression.Progress
			achievement.CompletedAt = dbAchievement.Progression.CompletedAtUTC
		}

		if dbAchievement.Enabled {
			achievements = append(achievements, achievement)
		}
	}

	return achievements
}
