package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"code.justin.tv/cb/achievements/internal/clients/db"
	"code.justin.tv/cb/achievements/internal/clients/stats"
	"code.justin.tv/cb/achievements/internal/mocks"
	"code.justin.tv/cb/achievements/view"
	"github.com/cactus/go-statsd-client/statsd"
	log "github.com/sirupsen/logrus"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type GetAchievementsSuite struct {
	suite.Suite

	reader *mocks.Reader
	users  *mocks.Users

	server           *Server
	channelID        string
	viewerID         string
	url              string
	request          *http.Request
	responseRecorder *httptest.ResponseRecorder
}

func (s *GetAchievementsSuite) SetupTest() {
	log.SetLevel(log.PanicLevel)

	s.reader = &mocks.Reader{}
	s.users = &mocks.Users{}
	statter, err := statsd.NewNoopClient()
	s.Nil(err)

	s.server = NewServer(&ServerParams{
		DBReader: s.reader,
		Users:    s.users,
		Statsd: &stats.Client{
			Statter: statter,
		},
	})

	s.channelID = "123456789"
	s.viewerID = "54321"
	s.url = fmt.Sprintf("/v1/internal/channel/%s/viewer/%s/achievements", s.channelID, s.viewerID)
	s.request = httptest.NewRequest(http.MethodGet, s.url, nil)
	s.responseRecorder = httptest.NewRecorder()
}

func (s *GetAchievementsSuite) TestBadRequest() {
	badChannelID := "NOT_NUMERIC"
	badViewerID := "abc"
	url := fmt.Sprintf("/v1/internal/channel/%s/viewer/%s/achievements", badChannelID, badViewerID)
	req := httptest.NewRequest(http.MethodGet, url, nil)

	s.server.ServeHTTP(s.responseRecorder, req)

	s.Equal(http.StatusBadRequest, s.responseRecorder.Code)
	s.Contains(s.responseRecorder.Body.String(), badChannelID)
}

func (s *GetAchievementsSuite) TestIsStaff() {
	s.users.On("IsAdmin", mock.Anything, s.viewerID).
		Return(true, nil)

	s.reader.On("SelectAllAchievementProgressionsForChannel", mock.Anything, s.channelID).
		Return(nil, nil)

	s.server.ServeHTTP(s.responseRecorder, s.request)

	s.Equal(http.StatusOK, s.responseRecorder.Code)
}

func (s *GetAchievementsSuite) TestUnauthorized() {
	s.users.On("IsAdmin", mock.Anything, s.viewerID).
		Return(false, nil)

	s.server.ServeHTTP(s.responseRecorder, s.request)

	s.Equal(http.StatusForbidden, s.responseRecorder.Code)
}

func (s *GetAchievementsSuite) TestInternalServerError_SelectAchievementsFailure() {
	s.reader.On("SelectAllAchievementProgressionsForChannel", mock.Anything, s.channelID).
		Return(nil, errors.New("failed to query"))
	s.users.On("IsAdmin", mock.Anything, s.viewerID).
		Return(true, nil)

	s.server.ServeHTTP(s.responseRecorder, s.request)

	s.Equal(http.StatusInternalServerError, s.responseRecorder.Code)
	s.Contains(s.responseRecorder.Body.String(), s.channelID)
}

func (s *GetAchievementsSuite) TestSuccess() {
	achievementID := "achievement id"
	progressCap := 9000
	level := 2
	img := "img.png"
	imgSm := "imgSm.png"
	img2x := "img2x.png"
	img3x := "img3x.png"
	updatedAt := time.Now().Add(-1 * time.Minute)
	now := time.Now()

	dbAchievements := []*db.Achievement{
		{
			ID:           achievementID,
			Key:          "achievement key",
			ProgressCap:  progressCap,
			Level:        level,
			Image:        img,
			ImageSm:      imgSm,
			Image2x:      img2x,
			Image3x:      img3x,
			Enabled:      true,
			CreatedAtUTC: time.Now().Add(-5 * time.Hour),
			UpdatedAtUTC: nil,

			Progression: &db.Progression{
				ID:             "progression id",
				ChannelID:      s.channelID,
				AchievementID:  achievementID,
				Progress:       progressCap,
				CreatedAtUTC:   time.Now().Add(-10 * time.Minute),
				UpdatedAtUTC:   &updatedAt,
				CompletedAtUTC: &now,
			},
		},
	}

	s.reader.On("SelectAllAchievementProgressionsForChannel", mock.Anything, s.channelID).
		Return(dbAchievements, nil)
	s.users.On("IsAdmin", mock.Anything, s.viewerID).
		Return(true, nil)

	s.server.ServeHTTP(s.responseRecorder, s.request)

	s.Equal(http.StatusOK, s.responseRecorder.Code)

	response := view.Achievements{}
	err := json.Unmarshal(s.responseRecorder.Body.Bytes(), &response)
	s.NoError(err)
	s.NotNil(response.Achievements)

	if s.Len(response.Achievements, len(dbAchievements)) {
		for i := 0; i < len(dbAchievements); i++ {
			achievement := response.Achievements[i]

			s.Equal(dbAchievements[i].ID, achievement.ID)
			s.Equal(dbAchievements[i].Key, achievement.Key)
			s.Equal(dbAchievements[i].Progression.Progress, achievement.Progress)
			s.Equal(dbAchievements[i].ProgressCap, achievement.ProgressCap)
			s.Equal(dbAchievements[i].Level, achievement.Level)
			s.Equal(dbAchievements[i].Image, achievement.Image)
			s.Equal(dbAchievements[i].ImageSm, achievement.ImageSm)
			s.Equal(dbAchievements[i].Image2x, achievement.Image2x)
			s.Equal(dbAchievements[i].Image3x, achievement.Image3x)

			if s.NotNil(achievement.CompletedAt) {
				s.True(dbAchievements[i].Progression.CompletedAtUTC.Equal(*achievement.CompletedAt))
			}
		}
	}
}

func (s *GetAchievementsSuite) TestDisabledAchievement() {
	achievementID := "achievement id"
	progressCap := 9000
	level := 2
	img := "img.png"
	imgSm := "imgSm.png"
	img2x := "img2x.png"
	img3x := "img3x.png"
	updatedAt := time.Now().Add(-1 * time.Minute)
	now := time.Now()

	dbAchievements := []*db.Achievement{
		{
			ID:           achievementID,
			Key:          "achievement key",
			ProgressCap:  progressCap,
			Level:        level,
			Image:        img,
			ImageSm:      imgSm,
			Image2x:      img2x,
			Image3x:      img3x,
			Enabled:      false,
			CreatedAtUTC: time.Now().Add(-5 * time.Hour),
			UpdatedAtUTC: nil,

			Progression: &db.Progression{
				ID:             "progression id",
				ChannelID:      s.channelID,
				AchievementID:  achievementID,
				Progress:       progressCap,
				CreatedAtUTC:   time.Now().Add(-10 * time.Minute),
				UpdatedAtUTC:   &updatedAt,
				CompletedAtUTC: &now,
			},
		},
	}
	s.reader.On("SelectAllAchievementProgressionsForChannel", mock.Anything, s.channelID).
		Return(dbAchievements, nil)
	s.users.On("IsAdmin", mock.Anything, s.viewerID).
		Return(true, nil)

	s.server.ServeHTTP(s.responseRecorder, s.request)

	s.Equal(http.StatusOK, s.responseRecorder.Code)

	response := view.Achievements{}
	err := json.Unmarshal(s.responseRecorder.Body.Bytes(), &response)

	s.NoError(err)
	s.Len(response.Achievements, 0)
}

func TestGetAchievementsSuite(t *testing.T) {
	suite.Run(t, &GetAchievementsSuite{})
}
