package api

import (
	"fmt"
	"net/http"

	"code.justin.tv/cb/achievements/internal/httputil"
)

func (s *Server) publish(w http.ResponseWriter, req *http.Request) {
	notification := req.URL.Query().Get("notification")

	switch notification {
	case "quest_complete":
		s.publishQuestComplete(w, req)
	case "achievement_complete":
		s.publishAchievementComplete(w, req)
	default:
		w.WriteHeader(http.StatusNotFound)
	}
}

func (s *Server) publishQuestComplete(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	quest := req.URL.Query().Get("quest")
	channelID := req.URL.Query().Get("channel_id")

	if len(quest) < 1 {
		writer.BadRequest("quest is required")
		return
	}

	if len(channelID) < 1 {
		writer.BadRequest("channel_id is required")
		return
	}

	err := s.notifier.PublishQuestComplete(req.Context(), quest, channelID)
	if err != nil {
		message := fmt.Sprintf("Failed to PublishQuestComplete for key (%s) and channel id (%s)", quest, channelID)
		writer.InternalServerError(message, err)
		return
	}

	w.WriteHeader(http.StatusAccepted)
}

func (s *Server) publishAchievementComplete(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	achievement := req.URL.Query().Get("achievement")
	channelID := req.URL.Query().Get("channel_id")

	if len(achievement) < 1 {
		writer.BadRequest("achievement is required")
		return
	}

	if len(channelID) < 1 {
		writer.BadRequest("channel_id is required")
		return
	}

	err := s.notifier.PublishAchievementComplete(req.Context(), achievement, channelID, 1)
	if err != nil {
		message := fmt.Sprintf("Failed to PublishAchievementComplete for achievement (%s) and channel id (%s)", achievement, channelID)
		writer.InternalServerError(message, err)
		return
	}

	w.WriteHeader(http.StatusAccepted)
}
