package api

import (
	"net/http"
	"time"

	"code.justin.tv/cb/achievements/internal/clients/ripley"

	"code.justin.tv/cb/achievements/internal/auth"
	"code.justin.tv/cb/achievements/internal/clients/db"
	"code.justin.tv/cb/achievements/internal/clients/dynamo"
	"code.justin.tv/cb/achievements/internal/clients/moneypenny"
	"code.justin.tv/cb/achievements/internal/clients/sns"
	"code.justin.tv/cb/achievements/internal/clients/stats"
	"code.justin.tv/cb/achievements/internal/clients/users"
	"code.justin.tv/cb/achievements/internal/httputil"
	"code.justin.tv/cb/achievements/internal/notifications"

	goji "goji.io"
	"goji.io/pat"
)

const handlerTimeout = 5 * time.Second

// Server contains a router, clients, and environment-dependent utilities.
type Server struct {
	mux        *goji.Mux
	auth       auth.Validator
	dbReader   db.Reader
	statsd     *stats.Client
	dynamoDB   dynamo.Dynamo
	sns        sns.SNS
	moneypenny moneypenny.Moneypenny
	notifier   notifications.Notifier
	users      users.Users
	ripley     ripley.Ripley
}

// ServerParams contains the required components of a Server.
type ServerParams struct {
	Auth       auth.Validator
	DBReader   db.Reader
	Statsd     *stats.Client
	DynamoDB   dynamo.Dynamo
	Sns        sns.SNS
	Moneypenny moneypenny.Moneypenny
	Notifier   notifications.Notifier
	Users      users.Users
	Ripley     ripley.Ripley
}

// NewServer instantiates a Server with the defined routes and corresponding handlers,
// and returns the Server.
func NewServer(params *ServerParams) *Server {
	server := &Server{
		mux:        goji.NewMux(),
		auth:       params.Auth,
		dbReader:   params.DBReader,
		statsd:     params.Statsd,
		dynamoDB:   params.DynamoDB,
		sns:        params.Sns,
		moneypenny: params.Moneypenny,
		notifier:   params.Notifier,
		users:      params.Users,
		ripley:     params.Ripley,
	}

	server.mux.Use(httputil.PanicRecoverer)
	server.mux.HandleFunc(pat.Get("/health"), httputil.HealthCheck)
	server.mux.HandleFunc(pat.Post("/notifications/publish"), server.publish)

	root := goji.SubMux()
	server.mux.Handle(pat.New("/*"), root)
	root.Use(httputil.AccessLogger)
	root.Use(server.statsd.StatsLogger)
	root.Use(httputil.Timeout(handlerTimeout))

	ChannelViewer := goji.SubMux()
	root.Handle(pat.New("/v1/internal/channel/:channel_id/viewer/:viewer_id/*"), ChannelViewer)
	ChannelViewer.Use(ValidateChannelID)
	ChannelViewer.Use(ValidateViewerID)
	ChannelViewer.HandleFunc(pat.Post("/streamManagerVisited"), server.v1StreamManagerVisited)
	ChannelViewer.HandleFunc(pat.Get("/achievements"), server.Achievements)

	return server
}

// ServeHTTP allows Server to implement http.Handler.
func (s *Server) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	s.mux.ServeHTTP(w, req)
}
