package api

import (
	"context"
	"fmt"
	"net/http"

	log "github.com/sirupsen/logrus"

	"code.justin.tv/cb/achievements/internal/clients/sns"
	"code.justin.tv/cb/achievements/internal/httputil"
	"code.justin.tv/cb/achievements/view"
)

// v1StreamManagerVisited responds to a POST request and pushes an SNS notification indicating that the
// Stream Manager has been visited
func (s *Server) v1StreamManagerVisited(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(ContextKeyChannelID).(string)
	viewerID := req.Context().Value(ContextKeyViewerID).(string)

	if channelID != viewerID {
		isAdmin, err := s.users.IsAdmin(req.Context(), viewerID)
		if err != nil {
			msg := fmt.Sprintf("api: failed to determine if viewer %s is an admin", viewerID)
			writer.InternalServerError(msg, err)
			return
		}

		if !isAdmin {
			msg := fmt.Sprintf("api: unauthorized to access stream-manager-visited for channel (ID %s) as viewer ID: %s", channelID, viewerID)
			writer.Forbidden(msg)
			return
		}
	}

	message := sns.NotificationMessage{
		ChannelID: channelID,
	}

	err := s.sns.PublishNotification(context.Background(), message)
	if err != nil {
		msg := fmt.Sprintf("api: failed to notify stream manager visited topic for channel_id: %s", channelID)
		log.WithError(err).WithFields(log.Fields{
			"channel": channelID,
		}).Error(msg)
		writer.InternalServerError(msg, err)
		return
	}

	writer.OK(&view.V1StreamManagerVisitedResponse{
		Status:  http.StatusOK,
		Message: "",
		Meta:    &view.V1StreamManagerVisitedMeta{},
		Data:    &view.V1StreamManagerVisitedData{},
	})
}
