package api

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/mock"

	"code.justin.tv/cb/achievements/internal/clients/sns"

	"code.justin.tv/cb/achievements/internal/clients/stats"
	"code.justin.tv/cb/achievements/internal/mocks"
	"github.com/cactus/go-statsd-client/statsd"
	log "github.com/sirupsen/logrus"
	"github.com/stretchr/testify/suite"
)

type v1StreamManagerVisitedSuite struct {
	suite.Suite

	reader     *mocks.Reader
	validator  *mocks.Validator
	dynamoDB   *mocks.Dynamo
	moneypenny *mocks.Moneypenny
	sns        *mocks.SNS
	users      *mocks.Users

	server           *Server
	channelID        string
	viewerID         string
	request          *http.Request
	responseRecorder *httptest.ResponseRecorder
}

func (s *v1StreamManagerVisitedSuite) SetupTest() {
	log.SetLevel(log.PanicLevel)

	s.validator = &mocks.Validator{}
	s.reader = &mocks.Reader{}
	s.dynamoDB = &mocks.Dynamo{}
	s.moneypenny = &mocks.Moneypenny{}
	s.sns = &mocks.SNS{}
	s.users = &mocks.Users{}
	statter, err := statsd.NewNoopClient()
	s.Nil(err)

	s.server = NewServer(&ServerParams{
		Auth:       s.validator,
		DBReader:   s.reader,
		DynamoDB:   s.dynamoDB,
		Moneypenny: s.moneypenny,
		Sns:        s.sns,
		Users:      s.users,
		Statsd: &stats.Client{
			Statter: statter,
		},
	})

	s.channelID = "123456"
	s.viewerID = "654321"
	s.request = httptest.NewRequest(http.MethodPost, fmt.Sprintf("/v1/internal/channel/%s/viewer/%s/streamManagerVisited", s.channelID, s.viewerID), nil)
	s.responseRecorder = httptest.NewRecorder()
}

func (s *v1StreamManagerVisitedSuite) TestBadChannelID() {
	badChannelID := "NOT_NUMERIC"
	url := fmt.Sprintf("/v1/internal/channel/%s/viewer/%s/streamManagerVisited", badChannelID, s.viewerID)
	req := httptest.NewRequest(http.MethodPost, url, nil)

	s.server.ServeHTTP(s.responseRecorder, req)

	s.Equal(http.StatusBadRequest, s.responseRecorder.Code)
	s.Contains(s.responseRecorder.Body.String(), badChannelID)
}

func (s *v1StreamManagerVisitedSuite) TestBadViewerID() {
	badViewerID := "NOT_NUMERIC"
	url := fmt.Sprintf("/v1/internal/channel/%s/viewer/%s/streamManagerVisited", s.channelID, badViewerID)
	req := httptest.NewRequest(http.MethodPost, url, nil)

	s.server.ServeHTTP(s.responseRecorder, req)

	s.Equal(http.StatusBadRequest, s.responseRecorder.Code)
	s.Contains(s.responseRecorder.Body.String(), badViewerID)
}

func (s *v1StreamManagerVisitedSuite) TestForbidden() {
	s.users.On("IsAdmin", mock.Anything, s.viewerID).
		Return(false, nil)

	s.server.ServeHTTP(s.responseRecorder, s.request)

	s.Equal(http.StatusForbidden, s.responseRecorder.Code)
	s.Contains(s.responseRecorder.Body.String(), s.channelID)
}

func (s *v1StreamManagerVisitedSuite) TestIntervalServerError_UsersFail() {
	s.users.On("IsAdmin", mock.Anything, s.viewerID).
		Return(false, errors.New("failed to retrieve user"))

	s.server.ServeHTTP(s.responseRecorder, s.request)

	s.Equal(http.StatusInternalServerError, s.responseRecorder.Code)
	s.Contains(s.responseRecorder.Body.String(), s.viewerID)
}

func (s *v1StreamManagerVisitedSuite) TestInternalServerError_PublishNotification() {
	s.users.On("IsAdmin", mock.Anything, s.viewerID).
		Return(true, nil)

	testMessage := sns.NotificationMessage{
		ChannelID: "123456",
	}
	s.sns.On("PublishNotification", context.Background(), testMessage).Return(errors.New("failed to publish notification"))

	s.server.ServeHTTP(s.responseRecorder, s.request)

	s.Equal(http.StatusInternalServerError, s.responseRecorder.Code)
	s.Contains(s.responseRecorder.Body.String(), s.channelID)
}

func (s *v1StreamManagerVisitedSuite) TestSuccess() {
	s.users.On("IsAdmin", mock.Anything, s.viewerID).
		Return(true, nil)

	testMessage := sns.NotificationMessage{
		ChannelID: "123456",
	}

	s.sns.On("PublishNotification", context.Background(), testMessage).Return(nil)
	s.server.ServeHTTP(s.responseRecorder, s.request)

	s.Equal(http.StatusOK, s.responseRecorder.Code)
}

func TestV1StreamManagerVisitedSuite(t *testing.T) {
	suite.Run(t, &v1StreamManagerVisitedSuite{})
}
