package auth

import (
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"code.justin.tv/common/goauthorization"
	"github.com/pkg/errors"
)

// Validator validates the capability claims of the
// "Twitch-Authorization" JWT in HTTP request headers.
type Validator interface {
	// ValidateProgressionViewing validates the capability "view_progressions"
	// to view a given channel's progressions.
	ValidateProgressionViewing(req *http.Request, channelID string) error
}

const (
	algorithm = "ES256"
	audience  = "code.justin.tv/cb/achievements"
	issuer    = "code.justin.tv/web/cartman"

	claimKeyViewProgressions = "view_progressions"

	paramKeyChannelID = "channel_id"
)

// Encoder is a wrapper for goauthorization.Encoder.
type Encoder struct {
	goauthorization.Encoder
}

// NewEncoder instantiates Encoder.
func NewEncoder(keyPath string) (*Encoder, error) {
	key, err := ioutil.ReadFile(keyPath)
	if err != nil {
		return nil, errors.Wrap(err, fmt.Sprintf("failed to read file at %s", keyPath))
	}

	encoder, err := goauthorization.NewEncoder(algorithm, issuer, key)
	if err != nil {
		return nil, errors.Wrap(err, "failed to instantiate encoder")
	}

	return &Encoder{encoder}, nil
}

// GenerateProgressionViewingToken instantiates an authorization token,
// with the capability "view_progressions" for a given channel ID,
// and returns the string encoding of the token.
func (e *Encoder) GenerateProgressionViewingToken(channelID string) (string, error) {
	claims := goauthorization.CapabilityClaims{
		claimKeyViewProgressions: goauthorization.CapabilityClaim{
			paramKeyChannelID: channelID,
		},
	}

	params := goauthorization.TokenParams{
		Exp:    time.Now().Add(900 * time.Second),
		Nbf:    time.Now(),
		Aud:    []string{audience},
		Claims: claims,
	}

	tokenString, err := e.Encode(params).String()
	if err != nil {
		return "", errors.Wrap(err, "failed to encode token to string")
	}

	return tokenString, nil
}

// Decoder is a wrapper for goauthorization.Decoder, implementing Validator.
type Decoder struct {
	goauthorization.Decoder
}

// NewDecoder instantiates Decoder.
func NewDecoder(keyPath string) (*Decoder, error) {
	key, err := ioutil.ReadFile(keyPath)
	if err != nil {
		return nil, errors.Wrap(err, fmt.Sprintf("failed to read file at %s", keyPath))
	}

	decoder, err := goauthorization.NewDecoder(algorithm, audience, issuer, key)
	if err != nil {
		return nil, errors.Wrap(err, "failed to instantiate decoder")
	}

	return &Decoder{decoder}, nil
}

// ValidateProgressionViewing validates that a given HTTP request contains
// a "Twitch-Authorization" token with the capability "view_progressions"
// to view a given channel's progressions.
func (d *Decoder) ValidateProgressionViewing(req *http.Request, channelID string) error {
	token, err := d.ParseToken(req)
	if err != nil {
		return errors.Wrap(err, "failed to parse token for progression viewing authorization")
	}

	claims := goauthorization.CapabilityClaims{
		claimKeyViewProgressions: goauthorization.CapabilityClaim{
			paramKeyChannelID: channelID,
		},
	}

	err = d.Validate(token, claims)
	if err != nil {
		return errors.Wrap(err, "failed to validate token for progression viewing authorization")
	}

	return nil
}
