package auth

import (
	"net/http"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
)

func TestNewEncoder_NoKey(t *testing.T) {
	bogusFilePath := "bogus"

	encoder, err := NewEncoder(bogusFilePath)
	assert.Contains(t, err.Error(), bogusFilePath)
	assert.Nil(t, encoder)
}

func TestNewEncoder_WrongKey(t *testing.T) {
	wrongFilePath := "development_ecc_public.key"

	encoder, err := NewEncoder(wrongFilePath)
	assert.Contains(t, err.Error(), "failed to instantiate encoder")
	assert.Nil(t, encoder)
}

func TestNewEncoder_Success(t *testing.T) {
	filePath := "development_ecc_private.key"

	encoder, err := NewEncoder(filePath)
	assert.NoError(t, err)
	assert.NotNil(t, encoder)
}

func TestNewDecoder_NoKey(t *testing.T) {
	bogusFilePath := "bogus"

	decoder, err := NewDecoder(bogusFilePath)
	assert.Contains(t, err.Error(), bogusFilePath)
	assert.Nil(t, decoder)
}

func TestNewDecoder_WrongKey(t *testing.T) {
	wrongFilePath := "development_ecc_private.key"

	decoder, err := NewDecoder(wrongFilePath)
	assert.Contains(t, err.Error(), "failed to instantiate decoder")
	assert.Nil(t, decoder)
}

func TestNewDecoder_Success(t *testing.T) {
	filePath := "development_ecc_public.key"

	decoder, err := NewDecoder(filePath)
	assert.NoError(t, err)
	assert.NotNil(t, decoder)
}

type progressionViewingSuite struct {
	suite.Suite
	encoder *Encoder
	decoder *Decoder
}

func TestProgressionViewingSuite(t *testing.T) {
	privateKeyPath := "development_ecc_private.key"
	publicKeyPath := "development_ecc_public.key"

	encoder, err := NewEncoder(privateKeyPath)
	require.NoError(t, err)
	require.NotNil(t, encoder)

	decoder, err := NewDecoder(publicKeyPath)
	require.NoError(t, err)
	require.NotNil(t, decoder)

	s := &progressionViewingSuite{
		encoder: encoder,
		decoder: decoder,
	}

	suite.Run(t, s)
}

func (s *progressionViewingSuite) TestTokenEncoding() {
	channelID := "test channel id"

	token, err := s.encoder.GenerateProgressionViewingToken(channelID)
	s.NoError(err)
	s.NotEmpty(token)

	req := &http.Request{
		Header: http.Header{},
	}
	req.Header.Set("Twitch-Authorization", token)

	err = s.decoder.ValidateProgressionViewing(req, channelID)
	s.NoError(err)
}

func (s *progressionViewingSuite) TestTokenParsingFailure() {
	channelID := "test channel id"

	token, err := s.encoder.GenerateProgressionViewingToken(channelID)
	s.NoError(err)
	s.NotEmpty(token)

	err = s.decoder.ValidateProgressionViewing(&http.Request{}, channelID)
	s.Contains(err.Error(), "failed to parse token")
}

func (s *progressionViewingSuite) TestTokenValidationFailure() {
	channelID := "test channel id"
	anotherChannelID := "some other channel id"

	token, err := s.encoder.GenerateProgressionViewingToken(channelID)
	s.NoError(err)
	s.NotEmpty(token)

	req := &http.Request{
		Header: http.Header{},
	}
	req.Header.Set("Twitch-Authorization", token)

	err = s.decoder.ValidateProgressionViewing(req, anotherChannelID)
	s.Contains(err.Error(), "failed to validate token")
}
