package dart

import (
	"context"
	"fmt"
	"strconv"

	receiver "code.justin.tv/amzn/TwitchDartReceiverTwirp"
	"code.justin.tv/cb/achievements/internal/notifications"
	"code.justin.tv/foundation/twitchclient"
)

type Client struct {
	dart receiver.Receiver
}

const (
	questCompleteNotificationType       = "quest_complete"
	achievementCompleteNotificationType = "achievement_complete"
	isTestSend                          = false
)

func (c Client) PublishAchievementComplete(ctx context.Context, key string, channelID string, progress int) error {
	badgeURL, ok := notifications.AchievementsBadgeURLMap[key]
	if !ok {
		return fmt.Errorf("dart: achievement key %s not found in badge url mapping", key)
	}

	metadata := map[string]string{
		"achievement":    key,
		"progress_level": convertProgressToString(progress),
		"badge_url":      badgeURL,
	}

	resp, err := c.dart.PublishNotification(ctx, &receiver.PublishNotificationRequest{
		NotificationType: achievementCompleteNotificationType,
		Recipient: &receiver.PublishNotificationRequest_RecipientId{
			RecipientId: channelID,
		},
		IsTestSend: isTestSend,
		Metadata:   metadata,
	})
	if err == nil {
		fmt.Printf("level=info op=PublishAchievementComplete channel_id=%s achievement=%s progress=%d trace_id=%s\n", channelID, key, progress, resp.GetNotificationTraceId())
	}

	return err
}

func (c Client) PublishQuestComplete(ctx context.Context, key string, channelID string) error {
	metadata := map[string]string{
		"quest_name": key,
	}

	resp, err := c.dart.PublishNotification(ctx, &receiver.PublishNotificationRequest{
		NotificationType: questCompleteNotificationType,
		Recipient: &receiver.PublishNotificationRequest_RecipientId{
			RecipientId: channelID,
		},
		IsTestSend: isTestSend,
		Metadata:   metadata,
	})
	if err == nil {
		fmt.Printf("level=info op=PublishQuestComplete channel_id=%s quest=%s trace_id=%s\n", channelID, key, resp.GetNotificationTraceId())
	}

	return err
}

func NewClient(host string) Client {
	client := receiver.NewReceiverProtobufClient(host, twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Host: host,
	}))

	return Client{
		dart: client,
	}
}

// Dirty way to convert progress levels to roman numerals. No need to handle anything
// greater than 5 since out achievement progression limit is 4.
func convertProgressToString(progress int) string {
	switch progress {
	case 0, 1:
		return "I"
	case 2:
		return "II"
	case 3:
		return "III"
	case 4:
		return "IV"
	case 5:
		return "V"
	default:
		return strconv.Itoa(progress)
	}
}
