package db

import (
	"database/sql"
	"fmt"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	driver             = "postgres"
	dataSourceTemplate = "postgres://%s@%s"
)

// Client is a wrapper for sql.DB.
type Client struct {
	*sql.DB
}

// Config includes required and optional parameters for OpenConnection.
type Config struct {
	Credentials  string
	Address      string
	MaxIdleConns int
}

// OpenConnection opens a connection pool to the PostgresSQL database,
// specified by the database credentials and address.
//
// Pinging the database ensures that the connection was established.
func OpenConnection(config Config) (*Client, error) {
	connection := fmt.Sprintf(dataSourceTemplate, config.Credentials, config.Address)

	session, err := sql.Open(driver, connection)
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to open connection")
	}

	if err := session.Ping(); err != nil {
		return nil, errors.Wrap(err, "db: failed to ping database")
	}

	if config.MaxIdleConns > 0 {
		session.SetMaxIdleConns(config.MaxIdleConns)
	}

	log.Info("Opened connection to Postgres database at ", config.Address)

	return &Client{session}, nil
}
