package db

import (
	"database/sql"
	"time"

	"github.com/lib/pq"
)

// Achievement represents an achievement type, each should map to
//	a unique job in the worker
type Achievement struct {
	ID          string
	Key         string
	ProgressCap int
	Level       int
	Image       string
	ImageSm     string
	Image2x     string
	Image3x     string
	Enabled     bool

	CreatedAtUTC time.Time
	UpdatedAtUTC *time.Time

	Progression *Progression
}

// IsStarted is used to determine whether an achievement has an associated
//	progression
func (a *Achievement) IsStarted() bool {
	return a.Progression != nil
}

// Progression represents a particular ChannelID's progress for a
//	particular achievement
type Progression struct {
	ID               string
	ChannelID        string
	AchievementID    string
	AchievementKey   string
	AchievementLevel int
	Progress         int

	CreatedAtUTC   time.Time
	UpdatedAtUTC   *time.Time
	CompletedAtUTC *time.Time
}

// IsComplete is used to determine whether a progression is completed
func (p *Progression) IsComplete() bool {
	return p.CompletedAtUTC != nil
}

// Progressions is a list type to add receviers on
type Progressions []Progression

// Completed gets completed progressions of a list of progressions
func (p Progressions) Completed() []Progression {
	completedProgressions := []Progression{}

	for _, progression := range p {
		if progression.IsComplete() {
			completedProgressions = append(completedProgressions, progression)
		}
	}
	return completedProgressions
}

// nullableProgression is strictly used for scanning a row,
// joined on progressions, that may not have a corresponding
// progression record (i.e. all selected progression fields are NULL).
type nullableProgression struct {
	ID            sql.NullString
	ChannelID     sql.NullString
	AchievementID sql.NullString
	Progress      sql.NullInt64

	CreatedAtUTC   pq.NullTime
	UpdatedAtUTC   pq.NullTime
	CompletedAtUTC pq.NullTime
}

func (nullable *nullableProgression) valid() bool {
	return nullable.ID.Valid &&
		nullable.ChannelID.Valid &&
		nullable.AchievementID.Valid &&
		nullable.Progress.Valid &&
		nullable.CreatedAtUTC.Valid
}

func (nullable *nullableProgression) toProgression(achievementLevel int) *Progression {
	if !nullable.valid() {
		return nil
	}

	progression := &Progression{
		ID:               nullable.ID.String,
		ChannelID:        nullable.ChannelID.String,
		AchievementLevel: achievementLevel,
		AchievementID:    nullable.AchievementID.String,
		Progress:         int(nullable.Progress.Int64),
		CreatedAtUTC:     nullable.CreatedAtUTC.Time,
	}

	if nullable.UpdatedAtUTC.Valid {
		progression.UpdatedAtUTC = &nullable.UpdatedAtUTC.Time
	}

	if nullable.CompletedAtUTC.Valid {
		progression.CompletedAtUTC = &nullable.CompletedAtUTC.Time
	}

	return progression
}

// AchievementGroup is a group of Achievements (same Key) sorted by ProgressCap.
type AchievementGroup []*Achievement

// NewChannelProgressions returns the completed and in-progress
// Progressions for a channel given a current progress.
func (ag AchievementGroup) NewChannelProgressions(channelID string, progress int) []Progression {
	progressions := []Progression{}
	completedAt := time.Now()

	for _, achievement := range ag {
		if progress < achievement.ProgressCap {
			progressions = append(progressions, Progression{
				ChannelID:        channelID,
				AchievementID:    achievement.ID,
				AchievementKey:   achievement.Key,
				AchievementLevel: achievement.Level,
				Progress:         progress,
				CompletedAtUTC:   nil,
			})

			continue
		}

		progressions = append(progressions, Progression{
			ChannelID:      channelID,
			AchievementID:  achievement.ID,
			Progress:       achievement.ProgressCap,
			CompletedAtUTC: &completedAt,
		})
	}

	return progressions
}
