package db

import (
	"context"
	"database/sql"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// SelectAchievementsByKey gets all achievements with the specified key
func (db *Client) SelectAchievementsByKey(ctx context.Context, key string) ([]*Achievement, error) {
	statement := `
	  SELECT
	  	a.id,
	  	a.key,
	  	a.progress_cap,
	  	a.img,
	  	a.img_sm,
	  	a.img_2x,
	  	a.img_3x,
	  	a.created_at_utc,
	  	a.updated_at_utc,
	  	a.enabled,
	  CASE WHEN l.num_levels > 1 THEN
	  	ROW_NUMBER() OVER (PARTITION BY a.key ORDER BY a.progress_cap ASC)
	  	ELSE 0
	  END AS level
	  FROM achievements as a
	  LEFT OUTER JOIN
	  	(
	  		SELECT
	  			key,
	  			COUNT(key) AS num_levels
	  		FROM achievements
	  		GROUP BY key
	  	) AS l
	  	ON l.key = a.key
	  WHERE a.key = $1
	  ORDER BY key, progress_cap ASC
	`

	var results []*Achievement

	rows, err := db.QueryContext(ctx, statement, key)

	switch {
	case err == sql.ErrNoRows:
		return results, nil
	case err != nil:
		return nil, errors.Wrap(err, "failed to query achievements by key")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).Error("db: failed to close pg rows")
		}
	}()

	for rows.Next() {
		achievement := &Achievement{}

		err = rows.Scan(
			&achievement.ID,
			&achievement.Key,
			&achievement.ProgressCap,
			&achievement.Image,
			&achievement.ImageSm,
			&achievement.Image2x,
			&achievement.Image3x,
			&achievement.CreatedAtUTC,
			&achievement.UpdatedAtUTC,
			&achievement.Enabled,
			&achievement.Level,
		)

		if err != nil {
			return nil, errors.Wrap(err, "failed to scan achievements rows")
		}

		results = append(results, achievement)
	}

	err = rows.Err()
	if err != nil {
		return nil, errors.Wrap(err, "failed to iterate achievement rows")
	}

	return results, nil
}
