package db

import (
	"context"
	"database/sql"
	"errors"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

type selectAchievementsByKeySuite struct {
	suite.Suite
	db   *Client
	mock sqlmock.Sqlmock
}

func TestSelectAchievementsByKeySuite(t *testing.T) {
	stub, mock, err := sqlmock.New()
	require.NoError(t, err)

	s := &selectAchievementsByKeySuite{
		db:   &Client{stub},
		mock: mock,
	}

	suite.Run(t, s)
}

func (s *selectAchievementsByKeySuite) TestEmpty() {
	key := "achievement key"

	s.mock.ExpectQuery("SELECT (.+) FROM achievements").WithArgs(key).WillReturnError(sql.ErrNoRows)

	achievements, err := s.db.SelectAchievementsByKey(context.Background(), key)
	s.NoError(err)
	s.Len(achievements, 0)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *selectAchievementsByKeySuite) TestQueryError() {
	key := "achievement key"
	queryError := errors.New("test query error")

	s.mock.ExpectQuery("SELECT (.+) FROM achievements").WithArgs(key).WillReturnError(queryError)

	achievements, err := s.db.SelectAchievementsByKey(context.Background(), key)
	s.Error(err)
	s.Nil(achievements)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *selectAchievementsByKeySuite) TestRowError() {
	key := "achievement key"

	rows := sqlmock.NewRows(
		[]string{"id", "key", "progress_cap", "created_at_utc", "updated_at_utc"},
	)

	rows = rows.AddRow("test id", key, 9000, time.Now(), nil).
		RowError(0, errors.New("test row error"))

	s.mock.ExpectQuery("SELECT (.+) FROM achievements").
		WithArgs(key).
		WillReturnRows(rows)

	achievements, err := s.db.SelectAchievementsByKey(context.Background(), key)
	s.Error(err)
	s.Nil(achievements)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *selectAchievementsByKeySuite) TestSuccess() {
	id := "achievement id"
	key := "achievement key"
	progressCap := 9000
	createdAt := time.Now()
	img := "imgurl"
	imgSm := "imgSmurl"
	img2x := "img2xurl"
	img3x := "img3xurl"
	enabled := true
	level := 1

	rows := sqlmock.NewRows(
		[]string{
			"id",
			"key",
			"progress_cap",
			"img",
			"img_sm",
			"img_2x",
			"img_3x",
			"created_at_utc",
			"updated_at_utc",
			"enabled",
			"level",
		},
	)

	rows = rows.AddRow(id, key, progressCap, img, imgSm, img2x, img3x, createdAt, nil, enabled, level)

	s.mock.ExpectQuery("SELECT (.+) FROM achievements").
		WithArgs(key).
		WillReturnRows(rows)

	achievements, err := s.db.SelectAchievementsByKey(context.Background(), key)
	s.NoError(err)

	if s.Len(achievements, 1) {
		s.Equal(id, achievements[0].ID)
		s.Equal(key, achievements[0].Key)
		s.Equal(progressCap, achievements[0].ProgressCap)
		s.Equal(img, achievements[0].Image)
		s.Equal(imgSm, achievements[0].ImageSm)
		s.Equal(img2x, achievements[0].Image2x)
		s.Equal(img3x, achievements[0].Image3x)
		s.Equal(enabled, achievements[0].Enabled)
		s.Equal(level, achievements[0].Level)
		s.Equal(createdAt, achievements[0].CreatedAtUTC)
		s.Nil(achievements[0].UpdatedAtUTC)
	}

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}
