package db

import (
	"context"
	"database/sql"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// SelectAllAchievementProgressionsForChannel gets all achievements,
// joined with a channel's progression (can be null).
func (db *Client) SelectAllAchievementProgressionsForChannel(ctx context.Context, channelID string) ([]*Achievement, error) {
	statement := `
		SELECT
			a.id,
			a.key,
			a.progress_cap,
			CASE WHEN l.num_levels > 1 THEN
				ROW_NUMBER() OVER (PARTITION BY a.key ORDER BY a.progress_cap ASC)
				ELSE 0
			END AS level,
			a.img,
			a.img_sm,
			a.img_2x,
			a.img_3x,
			a.created_at_utc,
			a.updated_at_utc,
			a.enabled,
			p.id,
			p.channel_id,
			p.achievement_id,
			p.progress,
			p.created_at_utc,
			p.updated_at_utc,
			p.completed_at_utc
		FROM achievements AS a
		LEFT OUTER JOIN
			(
				SELECT
				key,
				COUNT(key) AS num_levels
				FROM achievements
				GROUP BY key
			) AS l
			ON l.key = a.key
		LEFT OUTER JOIN
			(
				SELECT *
				FROM progressions
				WHERE channel_id = $1
			) AS p
			ON p.achievement_id = a.id
		ORDER BY
			p.completed_at_utc DESC NULLS LAST,
			p.progress DESC NULLS LAST,
			a.key ASC,
			level ASC
	`

	var achievements []*Achievement

	rows, err := db.QueryContext(ctx, statement, channelID)

	switch {
	case err == sql.ErrNoRows:
		return achievements, nil
	case err != nil:
		msg := "failed to query achievements with progression for channel"
		return nil, errors.Wrap(err, msg)
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).Error("db: failed to close pg rows")
		}
	}()

	for rows.Next() {
		achievement := &Achievement{}
		nullableProgression := &nullableProgression{}

		err = rows.Scan(
			&achievement.ID,
			&achievement.Key,
			&achievement.ProgressCap,
			&achievement.Level,
			&achievement.Image,
			&achievement.ImageSm,
			&achievement.Image2x,
			&achievement.Image3x,
			&achievement.CreatedAtUTC,
			&achievement.UpdatedAtUTC,
			&achievement.Enabled,
			&nullableProgression.ID,
			&nullableProgression.ChannelID,
			&nullableProgression.AchievementID,
			&nullableProgression.Progress,
			&nullableProgression.CreatedAtUTC,
			&nullableProgression.UpdatedAtUTC,
			&nullableProgression.CompletedAtUTC,
		)

		if err != nil {
			msg := "failed to scan rows for achievements with progression for channel"
			return nil, errors.Wrap(err, msg)
		}

		achievement.Progression = nullableProgression.toProgression(achievement.Level)
		achievements = append(achievements, achievement)
	}

	err = rows.Err()
	if err != nil {
		msg := "failed to iterate rows for achievements with progression for channel"
		return nil, errors.Wrap(err, msg)
	}

	return achievements, nil
}
