package db

import (
	"context"
	"database/sql"
	"errors"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	sqlmock "gopkg.in/DATA-DOG/go-sqlmock.v1"
)

type selectAllAchievementProgressionsForChannelSuite struct {
	suite.Suite
	db   *Client
	mock sqlmock.Sqlmock
}

func TestSelectAllAchievementProgressionsForChannelSuite(t *testing.T) {
	stub, mock, err := sqlmock.New()
	require.NoError(t, err)

	s := &selectAllAchievementProgressionsForChannelSuite{
		db:   &Client{stub},
		mock: mock,
	}

	suite.Run(t, s)
}

func (s *selectAllAchievementProgressionsForChannelSuite) TestEmpty() {
	channelID := "test channel id"

	s.mock.ExpectQuery("SELECT (.+) FROM achievements (.+) LEFT OUTER JOIN (.+) LEFT OUTER JOIN (.+)").
		WithArgs(channelID).
		WillReturnError(sql.ErrNoRows)

	achievements, err := s.db.SelectAllAchievementProgressionsForChannel(
		context.Background(),
		channelID,
	)

	s.NoError(err)
	s.Len(achievements, 0)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *selectAllAchievementProgressionsForChannelSuite) TestQueryError() {
	channelID := "test channel id"

	s.mock.ExpectQuery("SELECT (.+) FROM achievements (.+) LEFT OUTER JOIN (.+) LEFT OUTER JOIN (.+)").
		WithArgs(channelID).
		WillReturnError(errors.New("test query error"))

	achievements, err := s.db.SelectAllAchievementProgressionsForChannel(
		context.Background(),
		channelID,
	)

	s.Contains(err.Error(), "failed to query")
	s.Nil(achievements)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *selectAllAchievementProgressionsForChannelSuite) TestScanError() {
	channelID := "test channel id"

	rows := sqlmock.NewRows(
		[]string{
			"a.id",
			"a.key",
			"a.progress_cap",
			"a.img",
			"a.img_sm",
			"a.img_2x",
			"a.img_3x",
			"a.created_at_utc",
			"a.updated_at_utc",
			"a.enabled",
			"p.id",
			"p.channel_id",
			"p.achievement_id",
			"p.progress",
			"p.created_at_utc",
			"p.updated_at_utc",
			"p.completed_at_utc",
		},
	).AddRow(
		"achievement id",
		"a.key",
		"THIS IS THE WRONG DATA TYPE",
		"img.png",
		"img.png",
		"img.png",
		"img.png",
		time.Now(),
		nil,
		true,
		"p.id",
		channelID,
		"achievement id",
		1,
		time.Now(),
		time.Now(),
		time.Now(),
	)

	s.mock.ExpectQuery("SELECT (.+) FROM achievements (.+) LEFT OUTER JOIN (.+) LEFT OUTER JOIN (.+)").
		WithArgs(channelID).
		WillReturnRows(rows)

	achievements, err := s.db.SelectAllAchievementProgressionsForChannel(
		context.Background(),
		channelID,
	)

	s.Contains(err.Error(), "failed to scan")
	s.Nil(achievements)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *selectAllAchievementProgressionsForChannelSuite) TestRowError() {
	channelID := "test channel id"

	rows := sqlmock.NewRows(
		[]string{
			"a.id",
			"a.key",
			"a.progress_cap",
			"a.img",
			"a.img_sm",
			"a.img_2x",
			"a.img_3x",
			"a.created_at_utc",
			"a.updated_at_utc",
			"a.enabled",
			"p.id",
			"p.channel_id",
			"p.achievement_id",
			"p.progress",
			"p.created_at_utc",
			"p.updated_at_utc",
			"p.completed_at_utc",
		},
	).AddRow(
		"achievement id",
		"a.key",
		1,
		"img.png",
		"img.png",
		"img.png",
		"img.png",
		time.Now(),
		nil,
		true,
		"p.id",
		channelID,
		"achievement id",
		1,
		time.Now(),
		time.Now(),
		time.Now(),
	).RowError(0, errors.New("test row error"))

	s.mock.ExpectQuery("SELECT (.+) FROM achievements (.+) LEFT OUTER JOIN (.+) LEFT OUTER JOIN (.+)").
		WithArgs(channelID).
		WillReturnRows(rows)

	achievements, err := s.db.SelectAllAchievementProgressionsForChannel(
		context.Background(),
		channelID,
	)

	s.Contains(err.Error(), "failed to iterate")
	s.Nil(achievements)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *selectAllAchievementProgressionsForChannelSuite) TestSuccess() {
	channelID := "test channel id"
	achievementID := "test achievement id"
	achievementKey := "test achievement key"
	achievementProgressCap := 2
	achievementLevel := 1
	achievementImg := "img.png"
	achievementImgSm := "img-sm.png"
	achievementImg2x := "img-2x.png"
	achievementImg3x := "img-3x.png"
	achievementCreatedAt := time.Now().Add(-72 * time.Hour)
	achievementUpdatedAt := time.Now().Add(-36 * time.Hour)
	achievementEnabled := true
	progressionID := "test progression id"
	progressionProgress := 1
	progressionCreatedAt := time.Now().Add(-24 * time.Hour)
	progressionUpdatedAt := time.Now().Add(-12 * time.Hour)
	progressionCompletedAt := time.Now()

	rows := sqlmock.NewRows(
		[]string{
			"a.id",
			"a.key",
			"a.progress_cap",
			"a.level",
			"a.img",
			"a.img_sm",
			"a.img_2x",
			"a.img_3x",
			"a.created_at_utc",
			"a.updated_at_utc",
			"a.enabled",
			"p.id",
			"p.channel_id",
			"p.achievement_id",
			"p.progress",
			"p.created_at_utc",
			"p.updated_at_utc",
			"p.completed_at_utc",
		},
	).AddRow(
		achievementID,
		achievementKey,
		achievementProgressCap,
		achievementLevel,
		achievementImg,
		achievementImgSm,
		achievementImg2x,
		achievementImg3x,
		achievementCreatedAt,
		achievementUpdatedAt,
		achievementEnabled,
		progressionID,
		channelID,
		achievementID,
		progressionProgress,
		progressionCreatedAt,
		progressionUpdatedAt,
		progressionCompletedAt,
	).AddRow(
		achievementID,
		achievementKey,
		achievementProgressCap,
		achievementLevel,
		achievementImg,
		achievementImgSm,
		achievementImg2x,
		achievementImg3x,
		achievementCreatedAt,
		achievementUpdatedAt,
		achievementEnabled,
		nil,
		nil,
		nil,
		nil,
		nil,
		nil,
		nil,
	)

	s.mock.ExpectQuery("SELECT (.+) FROM achievements (.+) LEFT OUTER JOIN (.+) LEFT OUTER JOIN (.+)").
		WithArgs(channelID).
		WillReturnRows(rows)

	achievements, err := s.db.SelectAllAchievementProgressionsForChannel(
		context.Background(),
		channelID,
	)
	s.NoError(err)

	if s.Len(achievements, 2) {
		a1 := *achievements[0]

		s.Equal(achievementID, a1.ID)
		s.Equal(achievementKey, a1.Key)
		s.Equal(achievementProgressCap, a1.ProgressCap)
		s.Equal(achievementLevel, a1.Level)
		s.Equal(achievementImg, a1.Image)
		s.Equal(achievementImgSm, a1.ImageSm)
		s.Equal(achievementImg2x, a1.Image2x)
		s.Equal(achievementImg3x, a1.Image3x)
		s.Equal(achievementCreatedAt, a1.CreatedAtUTC)

		if s.NotNil(a1.UpdatedAtUTC) {
			s.Equal(achievementUpdatedAt, *a1.UpdatedAtUTC)
		}

		if s.NotNil(a1.Progression) {
			progression := a1.Progression

			s.Equal(progressionID, progression.ID)
			s.Equal(channelID, progression.ChannelID)
			s.Equal(achievementID, progression.AchievementID)
			s.Equal(progressionProgress, progression.Progress)
			s.Equal(progressionCreatedAt, progression.CreatedAtUTC)

			if s.NotNil(progression.UpdatedAtUTC) {
				s.Equal(progressionUpdatedAt, *progression.UpdatedAtUTC)
			}

			if s.NotNil(progression.CompletedAtUTC) {
				s.Equal(progressionCompletedAt, *progression.CompletedAtUTC)
			}
		}

		a2 := *achievements[1]

		s.Equal(achievementID, a2.ID)
		s.Equal(achievementKey, a2.Key)
		s.Equal(achievementProgressCap, a2.ProgressCap)
		s.Equal(achievementLevel, a2.Level)
		s.Equal(achievementImg, a1.Image)
		s.Equal(achievementImgSm, a1.ImageSm)
		s.Equal(achievementImg2x, a1.Image2x)
		s.Equal(achievementImg3x, a1.Image3x)
		s.Equal(achievementCreatedAt, a2.CreatedAtUTC)

		if s.NotNil(a2.UpdatedAtUTC) {
			s.Equal(achievementUpdatedAt, *a2.UpdatedAtUTC)
		}

		s.Nil(a2.Progression)
	}

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}
