package db

import (
	"context"
	"database/sql"
	"fmt"
	"strings"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	namedParamterTemplate = "($%d, $%d::uuid, $%d::integer, $%d::timestamp)"
)

// UpsertProgressions updates the specified progression's
// progress and completed_at_utc values.
//
// All upserted Progressions are returned.
func (db *Client) UpsertProgressions(ctx context.Context, params []Progression, progressRelapsable bool) ([]Progression, error) {
	if len(params) == 0 {
		return nil, nil
	}

	values := []interface{}{}
	for _, param := range params {
		values = append(values, param.ChannelID, param.AchievementID, param.Progress, param.CompletedAtUTC)
	}

	namedParameters := buildNamedParamterForUpsert(len(values) / 4)

	var updateProgressClause string
	if progressRelapsable {
		updateProgressClause = "EXCLUDED.progress"
	} else {
		updateProgressClause = `
      (
        SELECT
          GREATEST(EXCLUDED.progress, progress)
        FROM
          progressions
        WHERE
          channel_id = EXCLUDED.channel_id
        AND
          achievement_id = EXCLUDED.achievement_id
      )
		`
	}

	statement := fmt.Sprintf(`
    WITH
      new_progressions (
        channel_id,
        achievement_id,
        progress,
        completed_at_utc
      ) AS (
        values %s
      ),
      incomplete_progressions AS (
        SELECT
          new_progressions.channel_id,
          new_progressions.achievement_id,
          new_progressions.progress,
          new_progressions.completed_at_utc
        FROM
          new_progressions
        LEFT JOIN
          progressions
        ON
          new_progressions.channel_id = progressions.channel_id
        AND
          new_progressions.achievement_id = progressions.achievement_id
        WHERE
          progressions.completed_at_utc IS NULL
      ),
      updated AS (
        INSERT INTO progressions (
          channel_id,
          achievement_id,
          progress,
          completed_at_utc
        )	(
          SELECT
            channel_id,
            achievement_id,
            progress,
            completed_at_utc
          FROM
            incomplete_progressions
        )
        ON CONFLICT (
          channel_id,
          achievement_id
        )
        DO UPDATE SET
          completed_at_utc = EXCLUDED.completed_at_utc,
          progress = %s
        RETURNING
          id,
          channel_id,
          achievement_id,
          progress,
          created_at_utc,
          updated_at_utc,
          completed_at_utc
      )
      SELECT
        updated.*,
        achievements.key
      FROM
        updated
      LEFT OUTER JOIN
        achievements ON updated.achievement_id = achievements.id
      ORDER BY
        channel_id,
        achievement_id,
        progress ASC
	`, namedParameters, updateProgressClause)

	results := []Progression{}
	rows, err := db.QueryContext(ctx, statement, values...)

	switch {
	case err == sql.ErrNoRows:
		return results, nil
	case err != nil:
		return nil, errors.Wrapf(err, "failed to upsert progressions with statement: %s, values: %+v", statement, values)
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).Error("db: failed to close pg rows")
		}
	}()

	for rows.Next() {
		progression := Progression{}

		err = rows.Scan(
			&progression.ID,
			&progression.ChannelID,
			&progression.AchievementID,
			&progression.Progress,
			&progression.CreatedAtUTC,
			&progression.UpdatedAtUTC,
			&progression.CompletedAtUTC,
			&progression.AchievementKey,
		)

		if err != nil {
			return nil, errors.Wrap(err, "failed to scan achievements rows")
		}

		results = append(results, progression)
	}

	if err != nil {
		return nil, errors.Wrap(err, "failed to update progression progress")
	}

	return results, nil
}

func buildNamedParamterForUpsert(numRows int) string {
	values := []string{}
	for i := 0; i < numRows; i++ {
		index := i * 4
		statement := fmt.Sprintf(namedParamterTemplate, index+1, index+2, index+3, index+4)
		values = append(values, statement)
	}
	return strings.Join(values, ", ")
}
