package db

import (
	"context"
	"errors"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	sqlmock "gopkg.in/DATA-DOG/go-sqlmock.v1"
)

type UpsertProgressionsSuite struct {
	suite.Suite
	db   *Client
	mock sqlmock.Sqlmock
}

func TestUpsertProgressionsSuite(t *testing.T) {
	stub, mock, err := sqlmock.New()
	require.NoError(t, err)

	s := &UpsertProgressionsSuite{
		db:   &Client{stub},
		mock: mock,
	}

	suite.Run(t, s)
}

func (s *UpsertProgressionsSuite) TestError() {
	id := "progression id"
	channelID := "channel id"
	achievementID := "4867b2fc-c713-451b-bd40-1af6668360f8"
	progress := 100
	now := time.Now().UTC()

	s.mock.ExpectQuery("INSERT INTO progressions").
		WithArgs(channelID, achievementID, progress, now).
		WillReturnError(errors.New("oops"))

	upserted, err := s.db.UpsertProgressions(context.Background(), []Progression{
		{
			ID:             id,
			AchievementID:  achievementID,
			ChannelID:      channelID,
			Progress:       progress,
			CompletedAtUTC: &now,
		},
	}, true)

	s.Error(err)
	s.Nil(upserted)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)
}

func (s *UpsertProgressionsSuite) TestRelapsableProgress() {
	id := "progression id"
	channelID := "channel id"
	achievementID := "4867b2fc-c713-451b-bd40-1af6668360f8"
	achievementKey := "achievement-key"
	progress := 100
	createdAt := time.Now().Add(time.Duration(-5))
	now := time.Now().UTC()

	progression := Progression{
		ID:             id,
		AchievementID:  achievementID,
		ChannelID:      channelID,
		Progress:       progress,
		CompletedAtUTC: &now,
	}

	s.mock.ExpectQuery(`
		INSERT INTO progressions
		(.+)
		DO UPDATE SET
			completed_at_utc = EXCLUDED.completed_at_utc,
			progress = \(
				SELECT GREATEST\(EXCLUDED.progress, progress\)
				FROM progressions
				WHERE channel_id = EXCLUDED.channel_id
				AND achievement_id = EXCLUDED.achievement_id
			\)
	`).
		WithArgs(
			progression.ChannelID,
			progression.AchievementID,
			progression.Progress,
			progression.CompletedAtUTC,
		).
		WillReturnRows(sqlmock.NewRows([]string{
			"id",
			"channel_id",
			"achievement_id",
			"progress",
			"created_at_utc",
			"updated_at_utc",
			"completed_at_utc",
			"achievement_key",
		}).AddRow(
			id,
			channelID,
			achievementID,
			progress,
			createdAt,
			now,
			now,
			achievementKey,
		))

	results, err := s.db.UpsertProgressions(context.Background(), []Progression{progression}, false)
	s.NoError(err)

	for _, upserted := range results {
		s.Equal(progression.ID, upserted.ID)
		s.Equal(progression.ChannelID, upserted.ChannelID)
		s.Equal(progression.AchievementID, upserted.AchievementID)
		s.Equal(progression.Progress, upserted.Progress)
		s.Equal(createdAt, upserted.CreatedAtUTC)
		s.Equal(achievementKey, upserted.AchievementKey)

		if s.NotNil(upserted.UpdatedAtUTC) {
			s.Equal(now, *upserted.UpdatedAtUTC)
		}

		if s.NotNil(upserted.CompletedAtUTC) {
			s.Equal(now, *upserted.CompletedAtUTC)
		}
	}
}

func (s *UpsertProgressionsSuite) TestSuccess() {
	id := "progression id"
	channelID := "channel id"
	achievementID1 := "4867b2fc-c713-451b-bd40-1af6668360f8"
	achievementID2 := "4867b2fc-c713-451b-bd40-1af6668360f8"
	achievementKeys := []string{"achievement-key-1", "achievement-key-2"}
	progress := 100
	createdAt := time.Now().Add(time.Duration(-5))
	now := time.Now().UTC()

	rows := sqlmock.NewRows(
		[]string{
			"id",
			"channel_id",
			"achievement_id",
			"progress",
			"created_at_utc",
			"updated_at_utc",
			"completed_at_utc",
			"achievement_key",
		},
	).AddRow(
		id,
		channelID,
		achievementID1,
		progress,
		createdAt,
		now,
		now,
		achievementKeys[0],
	).AddRow(
		id,
		channelID,
		achievementID2,
		progress,
		createdAt,
		now,
		nil,
		achievementKeys[1],
	)

	progression1 := Progression{
		ID:             id,
		AchievementID:  achievementID1,
		ChannelID:      channelID,
		Progress:       progress,
		CompletedAtUTC: &now,
	}
	progression2 := Progression{
		ID:             id,
		AchievementID:  achievementID2,
		ChannelID:      channelID,
		Progress:       progress,
		CompletedAtUTC: nil,
	}

	values := []Progression{
		progression1,
		progression2,
	}

	s.mock.ExpectQuery("INSERT INTO progressions").
		WithArgs(
			progression1.ChannelID,
			progression1.AchievementID,
			progression1.Progress,
			progression1.CompletedAtUTC,
			progression2.ChannelID,
			progression2.AchievementID,
			progression2.Progress,
			progression2.CompletedAtUTC,
		).
		WillReturnRows(rows)

	results, err := s.db.UpsertProgressions(context.Background(), values, true)

	s.NoError(err)

	for i, upserted := range results {
		s.Equal(values[i].ID, upserted.ID)
		s.Equal(values[i].ChannelID, upserted.ChannelID)
		s.Equal(values[i].AchievementID, upserted.AchievementID)
		s.Equal(values[i].Progress, upserted.Progress)
		s.Equal(createdAt, upserted.CreatedAtUTC)
		s.Equal(achievementKeys[i], upserted.AchievementKey)

		if s.NotNil(upserted.UpdatedAtUTC) {
			s.Equal(now, *upserted.UpdatedAtUTC)
		}

		if upserted.CompletedAtUTC != nil {
			s.NotNil(upserted.CompletedAtUTC)
			s.Equal(now, *upserted.CompletedAtUTC)
		} else {
			s.Nil(upserted.CompletedAtUTC)
		}
	}

	// Inserting one new completed progression should return only one row
	rows = sqlmock.NewRows(
		[]string{
			"id",
			"channel_id",
			"achievement_id",
			"progress",
			"created_at_utc",
			"upserted_at_utc",
			"completed_at_utc",
			"achievement_key",
		},
	).AddRow(
		id,
		channelID,
		achievementID2,
		progress,
		createdAt,
		now,
		now,
		achievementKeys[0],
	)

	progression2.CompletedAtUTC = &now
	values = []Progression{
		progression1,
		progression2,
	}

	s.mock.ExpectQuery("INSERT INTO progressions").
		WithArgs(
			progression1.ChannelID,
			progression1.AchievementID,
			progression1.Progress,
			progression1.CompletedAtUTC,
			progression2.ChannelID,
			progression2.AchievementID,
			progression2.Progress,
			progression2.CompletedAtUTC,
		).
		WillReturnRows(rows)
	results, err = s.db.UpsertProgressions(context.Background(), values, true)
	s.NoError(err)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)

	s.Equal(1, len(results))
	upserted := results[0]
	s.Equal(progression2.ID, upserted.ID)
	s.Equal(progression2.ChannelID, upserted.ChannelID)
	s.Equal(progression2.AchievementID, upserted.AchievementID)
	s.Equal(progression2.Progress, upserted.Progress)
	s.Equal(createdAt, upserted.CreatedAtUTC)
	s.Equal(now, *upserted.UpdatedAtUTC)
	s.Equal(now, *upserted.CompletedAtUTC)
	s.Equal(achievementKeys[0], upserted.AchievementKey)

	// Inserting again should expect no rows
	s.mock.ExpectQuery("INSERT INTO progressions").
		WithArgs(
			progression1.ChannelID,
			progression1.AchievementID,
			progression1.Progress,
			progression1.CompletedAtUTC,
			progression2.ChannelID,
			progression2.AchievementID,
			progression2.Progress,
			progression2.CompletedAtUTC,
		).
		WillReturnRows(sqlmock.NewRows([]string{}))
	results, err = s.db.UpsertProgressions(context.Background(), values, true)
	s.NoError(err)

	err = s.mock.ExpectationsWereMet()
	s.NoError(err)

	s.Equal(0, len(results))
}
