package dynamo

import (
	"context"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/cb/achievements/config"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/pkg/errors"
)

// ChannelUpdate is a struct containing the timestamps of when a channel
// made their first update to their community, game, and title
type ChannelUpdate struct {
	ChannelID          string     `json:"channel_id"`
	CommunityUpdatedAt *time.Time `json:"community_updated_at"`
	GameUpdatedAt      *time.Time `json:"game_updated_at"`
	TitleUpdatedAt     *time.Time `json:"title_updated_at"`
}

// TableHasChannel checks whether the specified channel has a row in the table
func (c *Client) TableHasChannel(ctx context.Context, table string, channelID string) (bool, error) {
	keyCondition := aws.String("channel_id = :channelID")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(channelID),
		},
	}

	results, err := c.dynamoDB.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(table),
		Limit:                     aws.Int64(1),
		ScanIndexForward:          aws.Bool(true),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
	})
	if err != nil {
		msg := fmt.Sprintf("dyamodb: failed to query from table %s for channel id %s", table, channelID)

		return false, errors.Wrap(err, msg)
	}

	return len(results.Items) > 0, nil
}

// PersistChannelToTable saves a row on the table for the specified channel
func (c *Client) PersistChannelToTable(ctx context.Context, table string, channelID string) error {
	now := time.Now()
	put := &dynamodb.PutItemInput{
		TableName: aws.String(table),
		Item: map[string]*dynamodb.AttributeValue{
			"channel_id": {S: aws.String(channelID)},
			"time":       {S: aws.String(now.UTC().Format(time.RFC3339))},
		},
	}

	_, err := c.dynamoDB.PutItemWithContext(ctx, put)
	if err != nil {
		msg := fmt.Sprintf("dyamodb: failed to update table %s for channel id %s", table, channelID)

		return errors.Wrap(err, msg)
	}

	return nil
}

// GetChannelUpdate finds a channelID and returns an array of strings
// that exist in the dynamodb row
func (c *Client) GetChannelUpdate(ctx context.Context, channelID string) (*ChannelUpdate, error) {
	channelUpdate := &ChannelUpdate{}
	table := config.Values.DynamoDB.Tables.ChannelUpdates
	keyCondition := aws.String("channel_id = :channelID")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(channelID),
		},
	}

	results, err := c.dynamoDB.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(table),
		Limit:                     aws.Int64(1),
		ScanIndexForward:          aws.Bool(true),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
	})
	if err != nil {
		msg := fmt.Sprintf("dynamodb: failed to query from table %s for channel id %s", table, channelID)

		return nil, errors.Wrap(err, msg)
	}
	if len(results.Items) < 1 {
		channelUpdate.ChannelID = channelID

		return channelUpdate, nil
	}

	err = dynamodbattribute.UnmarshalMap(results.Items[0], channelUpdate)
	if err != nil {
		msg := fmt.Sprintf("dynamodb: failed to unmarshal results from table %s for channel id %s", table, channelID)

		return nil, errors.Wrap(err, msg)
	}

	return channelUpdate, nil
}

// SaveChannelUpdate saves a preformatted row containing a channel_id and
// other attributes onto the target table
func (c *Client) SaveChannelUpdate(ctx context.Context, channelUpdate *ChannelUpdate) error {
	table := config.Values.DynamoDB.Tables.ChannelUpdates

	expressionAttributeNames := map[string]*string{}
	expressionAttributeValues := map[string]*dynamodb.AttributeValue{}
	updateExpression := []string{}

	// if channelUpdate is empty, then don't make call to dynamodb
	if channelUpdate.CommunityUpdatedAt == nil && channelUpdate.GameUpdatedAt == nil && channelUpdate.TitleUpdatedAt == nil {
		return nil
	}

	if channelUpdate.CommunityUpdatedAt != nil {
		expressionAttributeNames["#C"] = aws.String("community_updated_at")
		expressionAttributeValues[":c"] = &dynamodb.AttributeValue{
			S: aws.String(channelUpdate.CommunityUpdatedAt.Format(time.RFC3339)),
		}
		updateExpression = append(updateExpression, " #C = :c")
	}

	if channelUpdate.GameUpdatedAt != nil {
		expressionAttributeNames["#G"] = aws.String("game_updated_at")
		expressionAttributeValues[":g"] = &dynamodb.AttributeValue{
			S: aws.String(channelUpdate.GameUpdatedAt.Format(time.RFC3339)),
		}
		updateExpression = append(updateExpression, " #G = :g")
	}

	if channelUpdate.TitleUpdatedAt != nil {
		expressionAttributeNames["#T"] = aws.String("title_updated_at")
		expressionAttributeValues[":t"] = &dynamodb.AttributeValue{
			S: aws.String(channelUpdate.TitleUpdatedAt.Format(time.RFC3339)),
		}
		updateExpression = append(updateExpression, " #T = :t")
	}

	updateExpressionString := fmt.Sprintf("SET%s", strings.Join(updateExpression, ","))

	_, err := c.dynamoDB.UpdateItemWithContext(ctx, &dynamodb.UpdateItemInput{
		TableName: aws.String(table),
		Key: map[string]*dynamodb.AttributeValue{
			"channel_id": {S: aws.String(channelUpdate.ChannelID)},
		},
		ExpressionAttributeNames:  expressionAttributeNames,
		ExpressionAttributeValues: expressionAttributeValues,
		UpdateExpression:          aws.String(updateExpressionString),
	})

	if err != nil {
		return errors.Wrap(err, "dynamodb: failed to update table item")
	}

	return nil
}
