package dynamo

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/cb/achievements/internal/awscredentials"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/pkg/errors"

	log "github.com/sirupsen/logrus"
)

const (
	// DynamoDBTimeFormat is the time format dynamo expects
	DynamoDBTimeFormat = "2006-01-02 15:04:05"

	// Maximum amount of write attempts before we give up trying
	// to write UnprocessedItems
	maxRetryAttempts = 3
)

// Client is a wrapper for the dynamodb api
type Client struct {
	env      string
	dynamoDB dynamodbiface.DynamoDBAPI
}

// NewClient creates an instance of a dynamoDB client
func NewClient(env string, region string) (*Client, error) {
	sess, err := session.NewSession(&aws.Config{
		Credentials: awscredentials.New(env, region),
		HTTPClient: &http.Client{
			Timeout: 3 * time.Second,
			Transport: &http.Transport{
				MaxIdleConnsPerHost: 200,
			},
		},
		MaxRetries: aws.Int(3),
		Region:     aws.String(region),
	})
	if err != nil {
		return nil, errors.Wrap(err, "dyamodb: failed to initialize dynamo db client")
	}

	return &Client{
		env:      env,
		dynamoDB: dynamodb.New(sess),
	}, nil
}

// sendBatchRequest handles sending a write batch to dynamo, with a backoff function
// to retry requests if there are any throttled write attempts
func (c *Client) sendBatchRequest(ctx context.Context, input *dynamodb.BatchWriteItemInput) error {
	// Dynamo may return unprocessed items in the response. To handle those, we retry those items
	// in a new batch, using a linear backoff to avoid excess throttling.
	// We limit the number of retries, and if we still have unprocessed items we log them.
	for attempts := 0; attempts < maxRetryAttempts; attempts++ {
		result, err := c.dynamoDB.BatchWriteItemWithContext(ctx, input)
		if err != nil {
			return err
		}

		if len(result.UnprocessedItems) == 0 {
			return nil
		}

		backoff := 2 * (attempts + 1)
		input = &dynamodb.BatchWriteItemInput{
			RequestItems: result.UnprocessedItems,
		}

		time.Sleep(time.Duration(backoff) * time.Second)
	}

	log.WithFields(log.Fields{
		"unprocessed_items": input.RequestItems,
	}).Error("dynamodb: max retries reached, but still found unprocessed items")

	return nil
}
