package dynamo

import (
	"context"
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/pkg/errors"
)

const (
	featureFlagTableName = "cb-achievements-feature_flags"

	// SendAchievementSNSFeature is the feature id for publishing to sns topic
	// whenever an achievement is completed
	SendAchievementSNSFeature = "SEND_ACHIEVEMENT_SNS_FEATURE"

	// InviteEligibleAffiliatesFeature is the feature id for sending an affiliate
	// invitation if a user is deemed eligible
	InviteEligibleAffiliatesFeature = "INVITE_ELIGIBLE_AFFILIATES_FEATURE"

	// NDaysNoViewersFeature is the feature id for enabling/disabling
	// the sourcing of this achievement. This is a legacy achievement
	// that was never enabled on production, but the code remains
	NDaysNoViewersFeature = "N_DAYS_NO_VIEWERS_FEATURE"

	// SpadeTrackingFeature is the feature id for enabling/disabling spending events
	// to spade during achievement completion. This can be disabled for performance
	// concerns
	SpadeTrackingFeature = "SPADE_TRACKING_FEATURE"

	// TalkALittleFeature is the feature is for enabling/disabling the Talk a Little achievement
	TalkALittleFeature = "TALK_A_LITTLE_FEATURE"
)

// Feature is the status for a feature of the achievements project
type Feature struct {
	FeatureID string `json:"feature_id"`
	Env       string `json:"env"`
	Enabled   bool   `json:"enabled"`
}

// IsFeatureEnabled checks whether a feature is enabled or not!
func (c *Client) IsFeatureEnabled(ctx context.Context, feature string) (bool, error) {
	keyCondition := aws.String("feature_id = :featureID AND env = :env")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":featureID": {
			S: aws.String(feature),
		},
		":env": {
			S: aws.String(c.env),
		},
	}

	results, err := c.dynamoDB.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(featureFlagTableName),
		Limit:                     aws.Int64(1),
		ScanIndexForward:          aws.Bool(true),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
	})
	if err != nil || len(results.Items) < 1 {
		msg := fmt.Sprintf("dyamodb: failed to find feature %s", feature)

		return false, errors.Wrap(err, msg)
	}

	unmarshalled := []Feature{}
	err = dynamodbattribute.UnmarshalListOfMaps(results.Items, &unmarshalled)
	if err != nil {
		msg := "dyamodb: failed to unmarshal feature query response"

		return false, errors.Wrap(err, msg)
	}

	return unmarshalled[0].Enabled, nil
}
